function [rxp, rxpS] = ...
  testRxReqEval(cfmRxAcqEnd, cfmRxHdr, cfmRxEnd)
%===============================================================================
%  Copyright (C) BAY9, 2014
%===============================================================================
%
% Syntax:
% [rxp, rxpS] = ...
%   testRxReqEval(cfmRxAcqEnd, cfmRxHdr, cfmRxEnd)
%
% Inputs:
%   cfmRxAcqEnd  - Message RxAcqEnd
%   cfmRxHdr     - Message RxHdr
%   cfmRxEnd     - Message RxEnd
%
% Outputs:
%   rxp  - Struct with RX results in a more readable form
%   rxpS - String to print all results in a row
%
% Description:
%   The function evaluates the RX confirm messages and returns parameters
%     rxp:
%      .acqStop:     0 = Frame synchronization, 1 = Acq stopped
%      .acqStopMode: 0 = Acq stopped normally after timeout, 
%                    1 = Error during call, RX not started
%      .hdrErr:      0 = Header OK, 1 = Header error
%      .mode:        0-7 = 6..54 MBit/s
%      .len:         RX data length in bytes
%      .fOffHz:      Frequency offset in Hz (for 312500Hz subcarrier spacing)
%      .amp:         RX baseband amplitude
%      .agcAttn:     RX attenuation setting
%      .crc:         0 = CRC ok, 1 = CRC error, 2 = header error, 3 = ACQ stop
%     rxpS: Similar outputs converted to strings
%        -> .hdrErr/.rxMode/.rxLen/.fOffHz/.agcAttn.crc
%       where some of the outputs are possibly replaced by '-' if not available.
%
  
  % ------------------------------------------------------------------------------
  % Global definitions
  global MsgIdWlan;
  
  %---------------------------------------------------------------------------
  % Evaluate parameters
  if (cfmRxAcqEnd(2) ~= 2)      % Mode 2 -> sync
    rxp.acqStop         = 1;
    rxp.acqStopMode     = cfmRxAcqEnd(2);
    rxp.crc             = 3;
  else
    rxp.acqStop         = 0;
    rxp.hdrErr          = cfmRxHdr(2);
    rxp.mode            = cfmRxHdr(3);
    rxp.len             = cfmRxHdr(4);
    fOff                = cfmRxHdr(5) - 2^16*(cfmRxHdr(5)>=2^15);
    rxp.fOffHz          = round(fOff / 2^13 * 312500);
    rxp.amp             = cfmRxHdr(6);
    rxp.agcAttn         = cfmRxHdr(7);
    rxp.frameStartTime  = cfmRxHdr(8) + 2^16*cfmRxHdr(9);

    if (rxp.hdrErr==0)
      rxp.crc       = cfmRxEnd(2);
    else
      rxp.crc       = 2;
    end
  end

  %---------------------------------------------------------------------------
  % Set corresponding strings
  if (rxp.acqStop)
    rxpS.hdrErr      = '-';
    rxpS.rxMode      = '-';
    rxpS.rxLen       = '  - ';
    rxpS.amp         = ' - ';
    rxpS.fOffHz      = '  ---  ';
    rxpS.agcAttn     = ' - ';
  else
    rxpS.hdrErr      = sprintf('%d',   rxp.hdrErr);
    rxpS.rxMode      = sprintf('%d',   rxp.mode);
    rxpS.rxLen       = sprintf('%4d',  rxp.len);
    rxpS.amp         = sprintf('%3d',  rxp.amp);
    rxpS.fOffHz      = sprintf('%+7d', rxp.fOffHz);
    rxpS.agcAttn     = sprintf('%3d',  rxp.agcAttn);
  end
  
  rxpS.crc = sprintf('%d', rxp.crc);

% ------------------------------------------------------------------------------
