function setupWlanAgc(id, attnDb, pinSet);
%===============================================================================
%  Copyright (C) BAY9, 2013
%===============================================================================
%
% Syntax:
%   setupWlanAgc(id, attnDb, pinSet)
%
% Inputs:
%   id      - Target identifier
%   attnDb  - Attenuation of the AGC steps in dB
%   pinSet  - Pin setting for the attenuation step
%
% Outputs:
%   --
%
% Description:
%   Prepare AGC depending on RF settings. The attenuation and corresponding pin
%   settings must be supplied highest to lowest gain. In order to calibrate the
%   AGC, the following steps are taken:
%     - Set AGC table using 70 pin setting supplied
%     - Set lower acquisition threshold
%     - Run through available gain steps to find proper input level
%     - Set final AGC table
%     - Set normal acquisition threshold again
%     - Set AGC to normal mode (real AGC, not fixed gain)
%
%===============================================================================

  % ------------------------------------------------------------------
  % Define messages
  global MsgIdWlan

  % --------------------------------------
  % RF calibration

  % If attn table has < 70 entries -> repeat the last one, otherwise
  % just use the first 70 entries
  attnTblLen = length(attnDb);
  if (attnTblLen < 70)
    pinSet70 = [pinSet repmat(pinSet(end), 1, 70-attnTblLen)];
  else
    pinSet70 = pinSet(1:70);
  end

  % Set initial RF attn table for calibration, real attenuation values
  % are not important here, only the pin settings
  fprintf('-------------------------------------\n');
  fprintf('Set RX AGC gain table for calibration\n');
  cfm = msgWlanCfgAgcTblReq(id, zeros(1, 70), pinSet70);

  % Set low ACQ threshold to produce false alarms -> measure the input amplitude
  fprintf('Set low ACQ threshold\n');
  cfm = msgWlanCfgAcqThrReq(id, -1);

  % Step through all gains, start with the highest, stop if
  % RX noise amplitude < 15 has been found
  fprintf('Send RX request messages 10x + evaluate input level\n');
  attnIdx = 0;
  while (attnIdx < 70)

    % Set attenuation manually
    fprintf('-------------------\n');
    fprintf('Set attnIdx -> %d\n', attnIdx);
    agcRunMode  = 1;
    agcInitAttn = attnIdx;
    cfm = msgWlanCfgAgcReq(id, agcRunMode, agcInitAttn);

    % Send RX request messages 10x + evaluate input level
    time32     = 40e6;    % A few seconds maximum
    timerMode  = 1;
    for k = 1:10
      fprintf('RX-ACQ run %d%c', k, char(13));
      [cfm, rxp] = msgWlanRxRun(id, timerMode, time32);
      amp(k) = rxp.amp;
    end

    % Calculate mean input level, end loop if < 15
    meanAmp(attnIdx+1) = mean(amp);
    fprintf('Input amplitude (mean) = %.1f\n', meanAmp(attnIdx+1));
    if (meanAmp(attnIdx+1) < 15)
      break;
    end

    attnIdx = attnIdx+1;

  end

  % Select the gain such that the average noise amplitude is > 15
  initAttnIdx = attnIdx - 1;

  % If meanAmp < 15 the first time -> nothing found -> error
  if (initAttnIdx == -1)
    fprintf('\nError, no attenuation with RX noise amplitude > 15\n');
    error();
  end

  fprintf('-------------------------------------\n');
  fprintf('Attenuation used: Index = %d, MeanAmp = %.1f\n', initAttnIdx, meanAmp(initAttnIdx+1));

  % Now the real attenuations are mapped to the AGC table, starting with the gain
  % found during calibration, this one has 0 dB attenuation by definition, all others
  % are chosen relative to it. Higher gains are not used.
  pinSetUsed  = pinSet(initAttnIdx+1:end);
  attnDbUsed  = attnDb(initAttnIdx+1:end) - attnDb(initAttnIdx+1);

  % Map real attenuation to closest values 0..69 dB
  for k = 0:69
    [minVal, minIdx]  = min(abs(k - attnDbUsed));
    attnDbLutF(k+1)   = attnDbUsed(minIdx);
    attnDbLut(k+1)    = round(attnDbUsed(minIdx));
    pinSetLut(k+1)    = pinSetUsed(minIdx);
  end

  % Set gain table gain based on calibration + mapping result
  fprintf('Set RX AGC gain table again after calibration\n');
  cfm = msgWlanCfgAgcTblReq(id, attnDbLut, pinSetLut);

  % DBG: Print AGC table finally
  if 0
    fprintf('\n');
    fprintf('-------------------------------------\n');
    fprintf(' Attn | AttnReal | PinSet\n');
    fprintf(' -----+----------+--------\n');
    for k = 0:69
      fprintf('  %2d  |   %4.1f   | %s \n', k, attnDbLutF(k+1), dec2bin(pinSetLut(k+1), 7));
    end
    fprintf('\n');
  end

  % Set normal ACQ threshold
  fprintf('-------------------------------------\n');
  fprintf('Set normal ACQ threshold + normal (automatic) AGC mode\n');
  cfm = msgWlanCfgAcqThrReq(id, 0);

  % Set automatic AGC
  agcRunMode  = 2;
  agcInitAttn = 0;
  cfm = msgWlanCfgAgcReq(id, agcRunMode, agcInitAttn);

  fprintf('-------------------------------------\n');

  % ------------------------------------------------------------------------------
