function setupWlan(id, cfg);
%===============================================================================
%  Copyright (C) BAY9, 2013
%===============================================================================
% function setupWlan:
%   Setup the 802.11a IP core
%
% Syntax:
%   setupWlan(id, cfg)
%
% Input:
%   id      - Target identifier
%   cfg     - config struct with field being used in message
%             see defaults in the source code for detailes
%
% Description:
%   This function prepares WLAN IP cores. In detail,
%     - Reset + boot
%     - Configure (use default values if cfg parameter is omitted)
%       o Zero-IF, no band reversal
%       o Shortest TX timing
%       o TX scaling = 14 dB backoff
%       o Time tracking parameters
%       o Set normal acquisition threshold
%       o Set normal (automatic) AGC
%
%===============================================================================

  % ------------------------------------------------------------------
  % Default idStd
  global idStd;
  if (isempty(idStd))
    idStd.tgt   = 'fpga0';
    idStd.top   = 'wlanX';
    idStd.inst  = 0;
  end

  % ------------------------------------------------------------------
  % Define connections
  if (nargin < 1)
    id = idStd;
  end

  % Default configuration
  if (nargin < 2)
    cfg = struct;
  end
    
  % ------------------------------------------------------------------
  %
  if (~isfield(cfg, 'ifsel'      )),  cfg.ifSel       = 0;        end   % Zero-IF
  if (~isfield(cfg, 'bandRev'    )),  cfg.bandRev     = 0;        end   % No band reversal
  if (~isfield(cfg, 'singleAdc'  )),  cfg.singleAdc   = 0;        end   % Use both ADCs

  if (~isfield(cfg, 'paDelay'    )),  cfg.paDelay     = 0;        end   % Start
  if (~isfield(cfg, 'txDelay'    )),  cfg.txDelay     = 0;        end   % everything
  if (~isfield(cfg, 'bbDelay'    )),  cfg.bbDelay     = 0;        end   % ASAP

  if (~isfield(cfg, 'txBackOffDb')),  cfg.txBackOffDb = 14;       end

  if (~isfield(cfg, 'fc    '     )),  cfg.fc          = 0;        end   % BB transmission
  if (~isfield(cfg, 'fSc   '     )),  cfg.fSc         = 312500;   end   % Subcarrier spacing
  if (~isfield(cfg, 'ttRev '     )),  cfg.ttRev       = 0;        end   % Normal time tracking

  if (~isfield(cfg, 'agcRunMode' )),  cfg.agcRunMode  = 2;        end   % Normal AGC mode
  if (~isfield(cfg, 'agcInitAttn')),  cfg.agcInitAttn = 0;        end   % Start with attn idx 0

  % ------------------------------------------------------------------
  % Define WLAN messages
  global MsgIdWlan
  def_MsgId_wlan;

  % ------------------------------------------------------------------
  % Reset + boot WLAN (SW reset is not needed if reset was applied externally)
  fprintf('-------------------------------------\n');
  fprintf('Boot WLAN\n');
  msgWlanResetReq(id);
  msgWlanBootReq(id);
  msgWlanVersionReq(id);
  msgWlanLedBlinkReq(id);

  % ------------------------------------------------------------------------------
  % Setup WLAN system
  fprintf('Set band selection\n');
  cfm = msgWlanCfgBandSelReq(id, cfg.ifSel, cfg.bandRev, cfg.singleAdc);

  fprintf('Set TX timing\n');
  cfm = msgWlanCfgTxTimingReq(id, cfg.paDelay, cfg.txDelay, cfg.bbDelay);

  fprintf('Set TX backoff\n');
  cfm = msgWlanCfgTxScalingReq(id, cfg.txBackOffDb);

  fprintf('Set RX time tracking parameters\n');
  cfm = msgWlanCfgTtReq(id, cfg.fc, cfg.fSc, cfg.ttRev);

  fprintf('Set normal ACQ threshold + normal (automatic) AGC mode\n');
  cfm = msgWlanCfgAcqThrReq(id, 0);
  cfm = msgWlanCfgAgcReq(id, cfg.agcRunMode, cfg.agcInitAttn);

  fprintf('Set CCA to 5500us (max length), 0 dB offset\n');
  cfm = msgWlanCfgCcaReq(id, 5500, 0);

  % ------------------------------------------------------------------------------
