//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   uart16
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module uart16(rxExtIf, rxExtIf_or, rxExtIf_ff, rxExtIf_we, 
              txExtIf, txExtIf_ir, txExtIf_fe, txExtIf_re, 
              txs, 
              rxs, 
              regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter rxExtIf_w         = 0;
  parameter txExtIf_w         = 0;

  parameter txs_w             = 0;
  parameter rxs_w             = 0;

  parameter div_w             = 0;
  parameter div_r             = 0;
  parameter div_s             = 0;
  parameter R_div             = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output      [rxExtIf_w-1:0]  rxExtIf;
  output                       rxExtIf_or;
  input                        rxExtIf_ff;
  output                       rxExtIf_we;
  assign                       rxExtIf_we = rxExtIf_or & ~rxExtIf_ff;

  input       [txExtIf_w-1:0]  txExtIf;
  output                       txExtIf_ir;
  input                        txExtIf_fe;
  output                       txExtIf_re;
  assign                       txExtIf_re = txExtIf_ir & ~txExtIf_fe;

  output          [txs_w-1:0]  txs;
  input           [rxs_w-1:0]  rxs;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire            [div_w-1:0]  div_q;
  wire            [div_w-1:0]  div_d;
  wire                         div_weint;
  assign                       div_weint = 0;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg div(div_q, regBus, div_d, div_weint, regWe, regRe, clk, reset);
  defparam div.w    = div_w;
  defparam div.rval = div_r;
  defparam div.sgn  = div_s;
  defparam div.adr  = R_div;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  //-----------------------------------------------------------------------------------------
  // Internal registers and variables
  reg                     resetD;                           // sampled reset signal
  
  // Definitions for RX part
  reg              [2:0]  rxState;
  reg              [0:0]  rxCoreState;
  reg             [18:0]  shRx;                             // RX input shift register
  reg              [3:0]  cnt0Rx;
  reg        [div_w-1:0]  cnt1Rx;                           // RX clk counter
  reg                     rx0, rx1;                         // sample the RX input line
  wire                    rxStart;                          // start signal for RX state machine

  // Definitions for TX part
  reg              [1:0]  txState;
  reg              [0:0]  txCoreState;
  reg             [18:0]  shTx;                             // TX output shift
  reg              [4:0]  cnt0Tx;
  reg        [div_w-1:0]  cnt1Tx;                           // TX clk counter
  
//=========================================================================================
  // Define parameters for main TX FSM
  parameter     StTxWaitNumLo = 0;  // Not needed anymore
  parameter     StTxWaitNumHi = 1;  // Not needed anymore
  parameter     StTxWaitData  = 2;
  parameter     StTxPassData  = 3;

  // Define parameters for core RX FSM
  parameter     StTxCoreWait  = 0;
  parameter     StTxCoreWrite = 1;

  // Define parameters for main RX FSM
  parameter     StRxWaitByte1 = 0;
  parameter     StRxReadByte1 = 1;
  parameter     StRxWaitByte2 = 2;
  parameter     StRxReadByte2 = 3;
  parameter     StRxPassData  = 4;

  // Define parameters for core RX FSM
  parameter     StRxCoreWait  = 0;
  parameter     StRxCoreRead  = 1;

  // TX assignments
  assign txExtIf_ir   = ~resetD & txState==StTxWaitData;
  assign txs          = shTx[0];
  
  // RX assignments
  assign  rxExtIf    = {shRx[17:10], shRx[7:0]};
  assign  rxExtIf_or = rxState==StRxPassData;
  assign  rxStart    = rx1 & ~rx0;

//----------------------------------------------------------------------------------------
//output state machine
  always @(posedge clk) begin

    // ----------------------------------------------------------------------------------------
    // Sample reset signal (remove ??)
    resetD   <= reset;
    
    // Main FSM - reset
    if (resetD) begin
      txState     <= StTxWaitData;
      txCoreState <= StTxCoreWait;
      shTx[0]     <= 1;
      rxState     <= StRxWaitByte1;
      rxCoreState <= StRxCoreWait;
      rx0         <= 0;
      rx1         <= 0;
    end else  
    
    // ----------------------------------------------------------------------------------------
    // Normal operation - if data is ready or does not need to be read
    begin

      // -----------------------------------------------------------------------
      // Sample RX input line
      rx0 <= rxs;
      rx1 <= rx0;
      
      // ---------------------------------------------------------------------------------
      // TX main FSM
      
      // Increase word counter -> write data
      if (txState==StTxWaitData) begin
        if (txExtIf_re) begin
          txState <= StTxPassData;
        end
      end else

      // Check if all words have been transmitted -> Done or wait for more data
      if (txState==StTxPassData) begin
        if (txCoreState==StTxCoreWait) begin
          txState <= StTxWaitData;
        end
      end
            
      //------------------------------------------------------------------------
      // Wait for start of transmission
      if (txCoreState==StTxCoreWait) begin                                  
        if (txState==StTxWaitData & txExtIf_re) begin
          txCoreState <= StTxCoreWrite;
          cnt0Tx      <= 20;                                  // reset counters
          cnt1Tx      <= div_q;
          shTx        <= {txExtIf[15:8], 1'b0, 1'b1, txExtIf[7:0], 1'b0};
        end
      end else
      
      // Write data via serial interface
      if (txCoreState==StTxCoreWrite) begin
        if (cnt1Tx!=0) begin
          cnt1Tx   <= cnt1Tx-1;                               // count down fast counter
        end else begin
          cnt1Tx   <= div_q;                                  // reset fast counter
          cnt0Tx   <= cnt0Tx-1;                               // cnt0Tx--

          if (cnt0Tx==0) begin                              // return to idle state
            txCoreState <= StTxCoreWait;
          end else begin
            shTx <= {1'b1, shTx[18:1]};
          end
        end // if (cnt1Tx!=0) .. else
      end
    
      // ---------------------------------------------------------------------------------
      // RX main FSM
      
      // Wait for 1st byte
      if (rxState==StRxWaitByte1) begin
        if (rxStart) begin
          rxState <= StRxReadByte1;
        end
      end else
      
      // Read 1st byte from serial port
      if (rxState==StRxReadByte1) begin
        if (rxCoreState==StRxCoreWait) begin
          rxState <= StRxWaitByte2;
        end
      end else
      
      // Wait for 2nd byte
      if (rxState==StRxWaitByte2) begin
        if (rxStart) begin
          rxState <= StRxReadByte2;
        end
      end else
      
      // Read 2nd byte from serial port
      if (rxState==StRxReadByte2) begin
        if (rxCoreState==StRxCoreWait) begin
          rxState <= StRxPassData;
        end
      end else

      // Pass data to output
      if (rxState==StRxPassData) begin
        rxState <= StRxWaitByte1;
      end 
      
      // -----------------------------------------------------------------------
      // RX core FSM
      
      // Wait for RX byte to start 
      if (rxCoreState==StRxCoreWait) begin                                  
        if (rxStart) begin
          rxCoreState <= StRxCoreRead;
          cnt0Rx      <= 9;                                     // reset counters
          cnt1Rx      <= {1'b0, div_q[div_w-1:1]};              // div / 2
        end
      end else
      
      // Get RX byte
      if (rxCoreState==StRxCoreRead) begin
        if (cnt1Rx!=0) begin
          cnt1Rx <= cnt1Rx-1;                               // count down fast counter
        end else begin
          cnt1Rx <= div_q;                                  // reset fast counter
          shRx   <= {rxs, shRx[18:1]};
          cnt0Rx <= cnt0Rx-1;                               // cnt0Rx--

          if (cnt0Rx==0) begin                              // return to idle state
            rxCoreState <= StRxCoreWait;
          end

        end // if (cnt1Rx!=0) .. else      
      end 
      
    end // if (run1_q==0) .. else
  end //always @(posedge clk)
  
endmodule
//=========================================================================================
