//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   sqrtX
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module sqrtX(irqDone, 
             regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters

  parameter irqDone_w         = 0;

  parameter run1_w            = 0;
  parameter run1_r            = 0;
  parameter run1_s            = 0;
  parameter R_run1            = 0;
  parameter xy_w              = 0;
  parameter xy_r              = 0;
  parameter xy_s              = 0;
  parameter R_xy              = 0;

  parameter w                 = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output      [irqDone_w-1:0]  irqDone;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire           [run1_w-1:0]  run1_q;
  wire           [run1_w-1:0]  run1_d;
  wire                         run1_weint;
  assign                       run1_weint = 0;

  wire             [xy_w-1:0]  xy_q;
  wire             [xy_w-1:0]  xy_d;
  wire                         xy_weint;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg run1(run1_q, regBus, run1_d, run1_weint, regWe, regRe, clk, reset);
  defparam run1.w    = run1_w;
  defparam run1.rval = run1_r;
  defparam run1.sgn  = run1_s;
  defparam run1.adr  = R_run1;

  ereg xy(xy_q, regBus, xy_d, xy_weint, regWe, regRe, clk, reset);
  defparam xy.w    = xy_w;
  defparam xy.rval = xy_r;
  defparam xy.sgn  = xy_s;
  defparam xy.adr  = R_xy;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Register definitions
  reg           startSqrt, startSqrtD;
  reg  [w-1:0]  x, y, z, m, x_nxt, y_nxt;
  reg    [w:0]  c;
  reg  [w/2:0]  m2;

  integer k;

  // --------------------------------------------------------------------------------------
  // Output register update and IRQ line
  assign xy_d       = y_nxt;
  assign xy_weint   = m2[0];
  assign irqDone    = m2[0];
  
  // --------------------------------------------------------------------------------------
  // Signal processing
  always @(*) begin
      
    // Avoid latches
    x_nxt = x;
    y_nxt = y;
    
    // Set m from m2 = {0, m2[w], 0, m2[w-1], 0, ..., 0, m2[1]}
    m = 0;
    for (k=0; k<w/2; k=k+1) begin
      m[2*k] = m2[k+1];
    end

    // Some calculations
    if (m2[0]==0) begin
      z = y + m;
      c = {1'b0, x} - {1'b0, z};
    end else begin
      c = {1'b0, y} - {1'b0, x};
      z = y + c[w];
    end

    // Init
    if (startSqrt) begin
      x_nxt  = regBus;
      y_nxt  = 0;
    end else

    // Main loop
    if (m2[0]==0) begin
      if (c[w]==0) begin
        x_nxt = c[w-1:0];
        y_nxt = (y>>1) + m;
      end else begin
        y_nxt = y>>1;
      end
    end else

    // Final rounding
    begin
      y_nxt = z;
    end
  
  end

  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin

    // Start signal
    startSqrt <= regBus==R_xy & regWe;

    // Processing updates
    x <= x_nxt;
    y <= y_nxt;

    // Reset if input is 0 (just in case...)
    if (run1_q==0) begin
      m2 <= 0;
    end else begin
    
      // Start processing
      if (startSqrt) begin
        m2 <= 1<<(w/2);
      end else
      
      // Shift >>1 in each step
      begin
        m2 <= m2>>1;    
      end

    end // if (run1_q==0)
  end // always

endmodule
//=========================================================================================

