//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   ppdu11a
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module ppdu11a(ys, ys_or, ys_ff, ys_we, 
               x, x_ir, x_fe, x_re, 
               irqHdrDone, 
               run1, 
               regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter ys_w              = 0;
  parameter x_w               = 0;

  parameter irqHdrDone_w      = 0;
  parameter run1_w            = 0;

  parameter plcpHdr0_w        = 0;
  parameter plcpHdr0_r        = 0;
  parameter plcpHdr0_s        = 0;
  parameter R_plcpHdr0        = 0;
  parameter plcpHdr1_w        = 0;
  parameter plcpHdr1_r        = 0;
  parameter plcpHdr1_s        = 0;
  parameter R_plcpHdr1        = 0;
  parameter numByte_w         = 0;
  parameter numByte_r         = 0;
  parameter numByte_s         = 0;
  parameter R_numByte         = 0;
  parameter numPadBit_w       = 0;
  parameter numPadBit_r       = 0;
  parameter numPadBit_s       = 0;
  parameter R_numPadBit       = 0;

  parameter StReset           = 0;
  parameter StHdr             = 0;
  parameter StHdrDone         = 0;
  parameter StPsduService     = 0;
  parameter StPsduData        = 0;
  parameter StPsduTail        = 0;
  parameter StPsduPad         = 0;
  parameter NumHdrData        = 0;
  parameter NumPsduService    = 0;
  parameter NumPsduTail       = 0;

  parameter state_w           = 0;
  parameter state_n           = 0;
  parameter state_m           = 0;
  parameter cnt_w             = 0;
  parameter cnt_n             = 0;
  parameter cnt_m             = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output           [ys_w-1:0]  ys;
  output                       ys_or;
  input                        ys_ff;
  output                       ys_we;
  assign                       ys_we = ys_or & ~ys_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  output   [irqHdrDone_w-1:0]  irqHdrDone;
  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire       [plcpHdr0_w-1:0]  plcpHdr0_q;
  wire       [plcpHdr0_w-1:0]  plcpHdr0_d;
  wire                         plcpHdr0_weint;

  wire       [plcpHdr1_w-1:0]  plcpHdr1_q;
  wire       [plcpHdr1_w-1:0]  plcpHdr1_d;
  wire                         plcpHdr1_weint;

  wire        [numByte_w-1:0]  numByte_q;
  wire        [numByte_w-1:0]  numByte_d;
  wire                         numByte_weint;
  assign                       numByte_weint = 0;

  wire      [numPadBit_w-1:0]  numPadBit_q;
  wire      [numPadBit_w-1:0]  numPadBit_d;
  wire                         numPadBit_weint;
  assign                       numPadBit_weint = 0;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg plcpHdr0(plcpHdr0_q, regBus, plcpHdr0_d, plcpHdr0_weint, regWe, regRe, clk, reset);
  defparam plcpHdr0.w    = plcpHdr0_w;
  defparam plcpHdr0.rval = plcpHdr0_r;
  defparam plcpHdr0.sgn  = plcpHdr0_s;
  defparam plcpHdr0.adr  = R_plcpHdr0;

  ereg plcpHdr1(plcpHdr1_q, regBus, plcpHdr1_d, plcpHdr1_weint, regWe, regRe, clk, reset);
  defparam plcpHdr1.w    = plcpHdr1_w;
  defparam plcpHdr1.rval = plcpHdr1_r;
  defparam plcpHdr1.sgn  = plcpHdr1_s;
  defparam plcpHdr1.adr  = R_plcpHdr1;

  ereg numByte(numByte_q, regBus, numByte_d, numByte_weint, regWe, regRe, clk, reset);
  defparam numByte.w    = numByte_w;
  defparam numByte.rval = numByte_r;
  defparam numByte.sgn  = numByte_s;
  defparam numByte.adr  = R_numByte;

  ereg numPadBit(numPadBit_q, regBus, numPadBit_d, numPadBit_weint, regWe, regRe, clk, reset);
  defparam numPadBit.w    = numPadBit_w;
  defparam numPadBit.rval = numPadBit_r;
  defparam numPadBit.sgn  = numPadBit_s;
  defparam numPadBit.adr  = R_numPadBit;

  // --------------------------------------------------------------------------------------
  // Internal status registers
  reg           [state_w-1:0]  state;
  reg             [cnt_w-1:0]  cnt;

  // --------------------------------------------------------------------------------------
  // Included instances
  `include "buf2_2.v"

  // --------------------------------------------------------------------------------------
  // Define connections
  assign ys                        = buf2_2_y;
  assign ys_or                     = buf2_2_y_or;
  assign buf2_2_y_ff               = ys_ff;


  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | buf2_2_regWe;
  assign regReOut = 0 | buf2_2_regRe;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Register definitions
  reg       [ys_w-1:0]  yOut;

  wire      [ys_w-1:0]  ysP;                     // Preliminary output
  wire                  ysP_we, ysP_or, ysP_ff;  // before buffering
  

  // --------------------------------------------------------------------------------------
  // Copy this autogenerated stuff to avoid non-causal use of ysP
  assign buf2_2_x                  = ysP;
  assign buf2_2_x_fe               = ~ysP_or;
  assign ysP_ff                    = ~buf2_2_x_ir;

  // --------------------------------------------------------------------------------------
  // Assign input and output enable
  assign x_ir   = cnt!=0 & (state==StPsduData & ysP_ff==0);
  assign ysP_or = cnt!=0 & (state!=StPsduData |   x_fe==0);
  assign ysP_we = ysP_or & ~ysP_ff;
  assign ysP    = yOut;
  
  // Output buffer control
  assign buf2_2_run1 = run1!=0;

  // Feed back the header input for shifting, use Ereg
  assign plcpHdr0_d     = {plcpHdr1_q[0], plcpHdr0_q[15:1]};
  assign plcpHdr1_d     = {1'b0, plcpHdr1_q[1]};
  assign plcpHdr0_weint = ysP_we & (run1==1);
  assign plcpHdr1_weint = ysP_we & (run1==1);

  // IRQ output - not used currently
  assign irqHdrDone = state==StHdr & cnt==1 & ysP_we; 

  // --------------------------------------------------------------------------------------
  // Signal processing behaviour
  always @(*) begin
        
    // Determine output y for header
    if (state==StHdr)
      yOut = {1'b0, plcpHdr0_q[0]};
    
    // Determine output for PSDU
    else if (state==StPsduService)     
      yOut = 2'b10;                
    else if (state==StPsduData)   
      yOut = {1'b1, x[0]};         
    else if (state==StPsduTail)   
      yOut = 2'b00;                
    else if (state==StPsduPad)    
      yOut = 2'b10;
    
    else
      yOut = 0;
    
  end

  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin
    
    // Reset
    if (run1==0) begin
      state <= StReset;
      cnt   <= 0;
    end else
      
    // Start transmission
    if (state==StReset) begin
      
      // -> Goto header transmission
      if (run1==1) begin
        state <= StHdr;
        cnt   <= NumHdrData;
      end else
      
      // Goto PSDU service transmission
      if (run1==2) begin
        state <= StPsduService;
        cnt   <= NumPsduService;
      end

    end else

    // Possibly goto PSDU directly from header -> simplify Matlab modelling
    if (state==StHdrDone) begin
      
      // Goto PSDU service transmission
      if (run1==2) begin
         state <= StPsduService;
       cnt   <= NumPsduService;
      end

    end else

    
    // Process transmission - continue if data is written to output
    if (ysP_we) begin

      // Count down first
      if (cnt!=1) begin
        cnt <= cnt-1;
      end else

      // Header -> end after 24 bit
      if (state==StHdr) begin
        state <= StHdrDone;
        cnt   <= 0;
      end else

      // PSDU: Service -> Data
      if (state==StPsduService) begin
        state <= StPsduData;
        cnt   <= {numByte_q, 3'b0};
      end else

      // PSDU: Data -> Tail
      if (state==StPsduData) begin
        state <= StPsduTail;
        cnt   <= NumPsduTail;
      end else
      
      // PSDU: Tail -> Pad
      if (state==StPsduTail) begin
        state <= StPsduPad;
        cnt   <= {6'b000000, numPadBit_q};
      end else 

      // PSDU: Pad -> End
      if (state==StPsduPad) begin
        cnt <= 0;
      end

    end
  end

endmodule
//=========================================================================================

