//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   irqCtrl
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module irqCtrl(irqOut, 
               irqIn, 
               regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters

  parameter irqOut_w          = 0;
  parameter irqIn_w           = 0;

  parameter run1_w            = 0;
  parameter run1_r            = 0;
  parameter run1_s            = 0;
  parameter R_run1            = 0;
  parameter irqStat_w         = 0;
  parameter irqStat_r         = 0;
  parameter irqStat_s         = 0;
  parameter R_irqStat         = 0;
  parameter irqEdgeLevel_w    = 0;
  parameter irqEdgeLevel_r    = 0;
  parameter irqEdgeLevel_s    = 0;
  parameter R_irqEdgeLevel    = 0;
  parameter irqMaskIn_w       = 0;
  parameter irqMaskIn_r       = 0;
  parameter irqMaskIn_s       = 0;
  parameter R_irqMaskIn       = 0;
  parameter irqMaskOut_w      = 0;
  parameter irqMaskOut_r      = 0;
  parameter irqMaskOut_s      = 0;
  parameter R_irqMaskOut      = 0;
  parameter irqClr_w          = 0;
  parameter irqClr_r          = 0;
  parameter irqClr_s          = 0;
  parameter R_irqClr          = 0;
  parameter irqSet_w          = 0;
  parameter irqSet_r          = 0;
  parameter irqSet_s          = 0;
  parameter R_irqSet          = 0;
  parameter irqIdx_w          = 0;
  parameter irqIdx_r          = 0;
  parameter irqIdx_s          = 0;
  parameter R_irqIdx          = 0;

  parameter w                 = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output       [irqOut_w-1:0]  irqOut;
  input         [irqIn_w-1:0]  irqIn;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire           [run1_w-1:0]  run1_q;
  wire           [run1_w-1:0]  run1_d;
  wire                         run1_weint;
  assign                       run1_weint = 0;

  wire        [irqStat_w-1:0]  irqStat_q;
  wire        [irqStat_w-1:0]  irqStat_d;
  wire                         irqStat_weint;

  wire   [irqEdgeLevel_w-1:0]  irqEdgeLevel_q;
  wire   [irqEdgeLevel_w-1:0]  irqEdgeLevel_d;
  wire                         irqEdgeLevel_weint;
  assign                       irqEdgeLevel_weint = 0;

  wire      [irqMaskIn_w-1:0]  irqMaskIn_q;
  wire      [irqMaskIn_w-1:0]  irqMaskIn_d;
  wire                         irqMaskIn_weint;
  assign                       irqMaskIn_weint = 0;

  wire     [irqMaskOut_w-1:0]  irqMaskOut_q;
  wire     [irqMaskOut_w-1:0]  irqMaskOut_d;
  wire                         irqMaskOut_weint;
  assign                       irqMaskOut_weint = 0;

  wire         [irqIdx_w-1:0]  irqIdx_q;
  wire         [irqIdx_w-1:0]  irqIdx_d;
  wire                         irqIdx_weint;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg run1(run1_q, regBus, run1_d, run1_weint, regWe, regRe, clk, reset);
  defparam run1.w    = run1_w;
  defparam run1.rval = run1_r;
  defparam run1.sgn  = run1_s;
  defparam run1.adr  = R_run1;

  ereg irqStat(irqStat_q, regBus, irqStat_d, irqStat_weint, regWe, regRe, clk, reset);
  defparam irqStat.w    = irqStat_w;
  defparam irqStat.rval = irqStat_r;
  defparam irqStat.sgn  = irqStat_s;
  defparam irqStat.adr  = R_irqStat;

  ereg irqEdgeLevel(irqEdgeLevel_q, regBus, irqEdgeLevel_d, irqEdgeLevel_weint, regWe, regRe, clk, reset);
  defparam irqEdgeLevel.w    = irqEdgeLevel_w;
  defparam irqEdgeLevel.rval = irqEdgeLevel_r;
  defparam irqEdgeLevel.sgn  = irqEdgeLevel_s;
  defparam irqEdgeLevel.adr  = R_irqEdgeLevel;

  ereg irqMaskIn(irqMaskIn_q, regBus, irqMaskIn_d, irqMaskIn_weint, regWe, regRe, clk, reset);
  defparam irqMaskIn.w    = irqMaskIn_w;
  defparam irqMaskIn.rval = irqMaskIn_r;
  defparam irqMaskIn.sgn  = irqMaskIn_s;
  defparam irqMaskIn.adr  = R_irqMaskIn;

  ereg irqMaskOut(irqMaskOut_q, regBus, irqMaskOut_d, irqMaskOut_weint, regWe, regRe, clk, reset);
  defparam irqMaskOut.w    = irqMaskOut_w;
  defparam irqMaskOut.rval = irqMaskOut_r;
  defparam irqMaskOut.sgn  = irqMaskOut_s;
  defparam irqMaskOut.adr  = R_irqMaskOut;

  ereg irqIdx(irqIdx_q, regBus, irqIdx_d, irqIdx_weint, regWe, regRe, clk, reset);
  defparam irqIdx.w    = irqIdx_w;
  defparam irqIdx.rval = irqIdx_r;
  defparam irqIdx.sgn  = irqIdx_s;
  defparam irqIdx.adr  = R_irqIdx;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Register definitions
  reg     [w-1:0]  irqInMD;
  reg              setWe, rstWe, re;
  wire    [w-1:0]  irqInM;
  reg       [3:0]  irqIdxD;
  wire     [16:0]  zeroExt;
  wire     [16:0]  irqStatQExt;
  
  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin
    
    // rst
    if (run1_q==0) begin
      irqInMD <= 0;
      setWe   <= 0;
      rstWe   <= 0;
      re      <= 0;
      
    // Normal operation
    end else begin
      
      setWe   <=  regWe & (regBus==R_irqSet);
      rstWe   <=  regWe & (regBus==R_irqClr);
      irqInMD <=  irqInM;
            
    end // if (run1_q==0) ... else
  end // always

  // Mask input IRQ
  assign irqInM = irqIn & irqMaskIn_q;
  
  // Set or rst IRQ status
  assign irqStat_d = (run1_q==0) ? 0 :
                     (irqInM & (~irqInMD | irqEdgeLevel_q))       | // Set via input       
                     (  {w{setWe}} & regBus[w-1:0])               | // Set via regBus      
                     (~({w{rstWe}} & regBus[w-1:0]) & irqStat_q);   // Reset via regBus    
  assign irqStat_weint = 1;

  // IRQ status -> Index
  assign zeroExt     = 0;
  assign irqStatQExt = {zeroExt[16-w:0], irqStat_q & irqMaskOut_q};
  always @(*)
    casex (irqStatQExt[15:0])
      16'bxxxxxxxxxxxxxxx1: irqIdxD =  0;
      16'bxxxxxxxxxxxxxx10: irqIdxD =  1;
      16'bxxxxxxxxxxxxx100: irqIdxD =  2;
      16'bxxxxxxxxxxxx1000: irqIdxD =  3;
      16'bxxxxxxxxxxx10000: irqIdxD =  4;
      16'bxxxxxxxxxx100000: irqIdxD =  5;
      16'bxxxxxxxxx1000000: irqIdxD =  6;
      16'bxxxxxxxx10000000: irqIdxD =  7;
      16'bxxxxxxx100000000: irqIdxD =  8;
      16'bxxxxxx1000000000: irqIdxD =  9;
      16'bxxxxx10000000000: irqIdxD = 10;
      16'bxxxx100000000000: irqIdxD = 11;
      16'bxxx1000000000000: irqIdxD = 12;
      16'bxx10000000000000: irqIdxD = 13;
      16'bx100000000000000: irqIdxD = 14;
      16'b1000000000000000: irqIdxD = 15;
      default:              irqIdxD = 0;
    endcase    
  assign irqIdx_d     = irqIdxD;
  assign irqIdx_weint = 1;
  
  // IRQ output line
  assign irqOut = |(irqStat_q & irqMaskOut_q);

endmodule
//=========================================================================================

