//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   intlv
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module intlv(y, y_or, y_ff, y_we, 
             x, x_ir, x_fe, x_re, 
             irqHdrDone, 
             run1, 
             regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter y_w               = 0;
  parameter x_w               = 0;

  parameter irqHdrDone_w      = 0;
  parameter run1_w            = 0;

  parameter mode1_w           = 0;
  parameter mode1_r           = 0;
  parameter mode1_s           = 0;
  parameter R_mode1           = 0;

  parameter cntOfdmWr_w       = 0;
  parameter cntOfdmWr_n       = 0;
  parameter cntOfdmWr_m       = 0;
  parameter cntOfdmRd_w       = 0;
  parameter cntOfdmRd_n       = 0;
  parameter cntOfdmRd_m       = 0;

  parameter mem0_w            = 0;
  parameter mem0_n            = 0;
  parameter mem0_m            = 0;
  parameter mem1_w            = 0;
  parameter mem1_n            = 0;
  parameter mem1_m            = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output            [y_w-1:0]  y;
  output                       y_or;
  input                        y_ff;
  output                       y_we;
  assign                       y_we = y_or & ~y_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  output   [irqHdrDone_w-1:0]  irqHdrDone;
  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire          [mode1_w-1:0]  mode1_q;
  wire          [mode1_w-1:0]  mode1_d;
  wire                         mode1_weint;
  assign                       mode1_weint = 0;

  // Wires for internal memories
  wire           [mem0_w-1:0]  mem0_q;
  wire           [mem0_w-1:0]  mem0_d;
  wire           [mem0_m-1:0]  mem0_adr;
  wire                         mem0_we;
  wire           [mem1_w-1:0]  mem1_q;
  wire           [mem1_w-1:0]  mem1_d;
  wire           [mem1_m-1:0]  mem1_adr;
  wire                         mem1_we;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg mode1(mode1_q, regBus, mode1_d, mode1_weint, regWe, regRe, clk, reset);
  defparam mode1.w    = mode1_w;
  defparam mode1.rval = mode1_r;
  defparam mode1.sgn  = mode1_s;
  defparam mode1.adr  = R_mode1;

  // --------------------------------------------------------------------------------------
  // Internal status registers
  reg       [cntOfdmWr_w-1:0]  cntOfdmWr;
  reg       [cntOfdmRd_w-1:0]  cntOfdmRd;

  // --------------------------------------------------------------------------------------
  // Internal single ported memories
  spRam1 mem0(mem0_q, mem0_d, mem0_adr, mem0_we, clk);
  defparam mem0.w    = mem0_w;
  defparam mem0.n    = mem0_n;
  defparam mem0.m    = mem0_m;

  spRam1 mem1(mem1_q, mem1_d, mem1_adr, mem1_we, clk);
  defparam mem1.w    = mem1_w;
  defparam mem1.n    = mem1_n;
  defparam mem1.m    = mem1_m;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

//-----------------------------------------------------------------------------------------

  // --------------------------------------------------------------------------
  // Define states, consider the 2 memories a 2 entry FIFO, copy from fifoA
  reg    [1:0]  memCnt;             // Fill level of the fifo 0..2
  reg           memRdAdr;           // Memory to be read currently
  reg           memWrAdr;           // Memory to be written currently       
  reg           memFe;              // Fifo empty    
  reg           memFf;              // Fifo full
  wire          memWe;              // } Write and read enable for FIFO
  wire          memRe;              // } Use when the last data is read/written
  reg           memWeD;             // Delayed write enable
  wire          wrMem0;             // Write to memory 0
  wire          wrMem1;             // Write to memory 1
  
                                       
  // Define registers and wires for interleaving and memory write
  reg         [8:0] cnt12, cnt12Nxt, cnt12P1;    // 
  reg         [8:0] cnt3,  cnt3Nxt;
  reg         [1:0] cnt4,  cnt4Nxt;
  reg         [1:0] cnt5,  cnt5Nxt;
  reg         [8:0] wrAdr, wrAdrNxt;
  reg         [1:0] modeQamWr;
  reg         [4:0] ncbps4Wr;           // Number of coded bit per symbol >> 4
  reg         [1:0] cntMax;             // Maximum for modulo counting
  reg         [8:0] cntMod;             // Counter modification

  // Define registers and wires for reading the memories
  reg         [8:0] rdAdr, rdAdrNxt, rdAdrP1;
  reg         [1:0] modeQamRd;
  reg         [4:0] ncbps4Rd;           // Number of coded bit per symbol >> 4

  // --------------------------------------------------------------------------
  // Define input and output control
  assign x_ir = run1 & (memFf==0);  // Input is ready if mem FIFO is not full
  assign y_or = run1 & (memFe==0);  // Output is ready if mem FIFO is not empty
  
  // --------------------------------------------------------------------------
  // Writing address calculation, create interleaving pattern
  always @(*) begin

    // Set BPSK for the 1st OFDM symbol
    modeQamWr = (cntOfdmWr==0) ? 0 : mode1_q;
  
    // ncbps4 = Number of coded bits per OFDM symbol >> 4
    case (modeQamWr)
      0: ncbps4Wr = 3;
      1: ncbps4Wr = 6;
      2: ncbps4Wr = 12;
      3: ncbps4Wr = 18;
    endcase

    // Maximum for the modulo counters
    case (modeQamWr)
      0: cntMax = 0;
      1: cntMax = 0;
      2: cntMax = 1;
      3: cntMax = 2;
    endcase

    // Count 0..ncbps-1
    cnt12P1 = cnt12+1;
    if (cnt12P1[8:4]==ncbps4Wr) begin
      cnt12Nxt = 0;
    end else begin
      cnt12Nxt = cnt12P1;
    end
    
    // Upon wrapping cnt1, cnt3 = 0, cnt4 = (cnt4+1)%s, cnt5 = 0
    if (cnt12Nxt[3:0]==0) begin
      cnt3Nxt = 0;
      if (cnt4==cntMax) begin
        cnt4Nxt = 0;
      end else begin
        cnt4Nxt = cnt4+1;
      end
      cnt5Nxt = 0;

    // Otherwise simply count up cnt3 = cnt3+q, cnt5 = (cnt5+1)%s
    end else begin
      cnt3Nxt  = cnt3+ncbps4Wr;
      cnt4Nxt  = cnt4;
      if (cnt5==cntMax) begin
        cnt5Nxt = 0;
      end else begin
        cnt5Nxt = cnt5+1;
      end
    end
    
     // Modification LUT
    case ({modeQamWr, cnt4Nxt, cnt5Nxt})
      6'b100001:  cntMod =  1;  // Table 2, idx 0,1     
      6'b100101:  cntMod = -1;  // Table 2, idx 1,1     Table 1: [0]
      6'b110001:  cntMod =  2;  // Table 3, idx 0,1     Table 2: [0  1]
      6'b110010:  cntMod =  1;  // Table 3, idx 0,2              [0 -1]
      6'b110101:  cntMod = -1;  // Table 3, idx 1,1     Table 3: [0  2  1]
      6'b110110:  cntMod =  1;  // Table 3, idx 1,2              [0 -1  1]
      6'b111001:  cntMod = -1;  // Table 3, idx 2,1              [0 -1 -2]
      6'b111010:  cntMod = -2;  // Table 3, idx 2,2
      default:    cntMod =  0;  // All other table entries  
    endcase

   // Finally, calculate next write address
    wrAdrNxt = (cnt12Nxt>>4) + cnt3Nxt + cntMod;
  end
  
  // Main FSM, only reset + feedback
  always @(posedge clk) begin
 
    // Reset
    if (run1==0) begin
      cnt12 <= 0;
      cnt3  <= 0;
      cnt4  <= 0;
      cnt5  <= 0;
      wrAdr <= 0;
    end else
    
    // Modify read address whenever data is read from input
    if (x_re) begin
      cnt12 <=  cnt12Nxt;
      cnt3  <=  cnt3Nxt;
      cnt4  <=  cnt4Nxt;
      cnt5  <=  cnt5Nxt;
      wrAdr <=  wrAdrNxt;
    end // if (run1==0) begin .. else
  end // always  
    
  // --------------------------------------------------------------------------
  // Define reading state machine - count address from 0..q*16-1
  always @(*) begin

    // Set BPSK for the 1st OFDM symbol
    modeQamRd = (cntOfdmRd==0) ? 0 : mode1_q;
  
    // ncbps4 = Number of coded bits per OFDM symbol >> 4
    case (modeQamRd)
      0: ncbps4Rd = 3;
      1: ncbps4Rd = 6;
      2: ncbps4Rd = 12;
      3: ncbps4Rd = 18;
    endcase

    // Count read address mod Ncbps
    rdAdrP1 = rdAdr+1;
    if (rdAdrP1[8:4]==ncbps4Rd) begin
      rdAdrNxt = 0;
    end else begin
      rdAdrNxt = rdAdrP1;
    end
  end
  
  // Reset and feedback
  always @(posedge clk) begin
    
    if (run1==0) begin
      rdAdr <= 0;
    end else
    
    if (y_we) begin
      rdAdr <= rdAdrNxt;
    end
  
  end

  // --------------------------------------------------------------------------  
  // Define buffer memory control
  assign wrMem0 = (memFf==0) & (memWrAdr==0);
  assign wrMem1 = (memFf==0) & (memWrAdr==1);

  assign mem0_we    = x_re & (memWrAdr==0);
  assign mem0_d     = x;
  assign mem0_adr   = wrMem0 ? wrAdr : (y_we ? rdAdrNxt : rdAdr);

  assign mem1_we    = x_re & (memWrAdr==1);
  assign mem1_d     = x;
  assign mem1_adr   = wrMem1 ? wrAdr : (y_we ? rdAdrNxt : rdAdr);
  
  // --------------------------------------------------------------------------
  // FIFO operation, FIFO is written/read when the last sample is written/read
  assign memWe = (cnt12P1[8:4]==ncbps4Wr) & x_re;
  assign memRe = (rdAdrP1[8:4]==ncbps4Rd) & y_we;

  always @(posedge clk) begin
  
  // Reset
    if (run1==0) begin
      memCnt    <= 0;
      memRdAdr  <= 0;
      memWrAdr  <= 0;
      memFe     <= 1;
      memFf     <= 0;
      memWeD    <= 0;
    end else  
    
    // Normal FIFO behaviour
    begin
      memWeD <= memWe;  // Use delayed indication that FIFO is not empty
      memFe  <= (memCnt==0 & !memWeD) | (memCnt==1 & memRe & !memWeD);
      memFf  <= (memCnt==2 & !memRe)  | (memCnt==1 & memWe & !memRe);
      memCnt <= memCnt+memWe-memRe;                                   
      if (memRe) begin
        memRdAdr <= memRdAdr+1;
      end    
      if (memWe) begin
        memWrAdr <= memWrAdr+1;
      end          
    end //normal operation
  end // always
  
  // --------------------------------------------------------------------------
  // Count OFDM symbol, 0 -> 1 if first symbol is written / read
  always @(posedge clk) begin
    
    if (run1==0) begin
      cntOfdmWr <= 0;
      cntOfdmRd <= 0;
    end else begin
    
      if (memWe) begin
        cntOfdmWr <= 1;
      end
      if (memRe) begin
        cntOfdmRd <= 1;
      end
      
    end
  end
  
  // --------------------------------------------------------------------------
  // Assign output y + IRQ
  assign y          = (memRdAdr==0) ? mem0_q : mem1_q;
  assign irqHdrDone = cntOfdmWr==0 & memWe==1;
 
 // --------------------------------------------------------------------------


endmodule
//=========================================================================================
