//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   fifoIf
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module fifoIf(tx, tx_or, tx_ff, tx_we, 
              rx, rx_ir, rx_fe, rx_re, 
              irqRx, 
              irqTx, 
              regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter tx_w              = 0;
  parameter rx_w              = 0;

  parameter irqRx_w           = 0;
  parameter irqTx_w           = 0;

  parameter run1_w            = 0;
  parameter run1_r            = 0;
  parameter run1_s            = 0;
  parameter R_run1            = 0;
  parameter rxDat_w           = 0;
  parameter rxDat_r           = 0;
  parameter rxDat_s           = 0;
  parameter R_rxDat           = 0;
  parameter rxSt_w            = 0;
  parameter rxSt_r            = 0;
  parameter rxSt_s            = 0;
  parameter R_rxSt            = 0;
  parameter txDat_w           = 0;
  parameter txDat_r           = 0;
  parameter txDat_s           = 0;
  parameter R_txDat           = 0;
  parameter txSt_w            = 0;
  parameter txSt_r            = 0;
  parameter txSt_s            = 0;
  parameter R_txSt            = 0;

  parameter w                 = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output           [tx_w-1:0]  tx;
  output                       tx_or;
  input                        tx_ff;
  output                       tx_we;
  assign                       tx_we = tx_or & ~tx_ff;

  input            [rx_w-1:0]  rx;
  output                       rx_ir;
  input                        rx_fe;
  output                       rx_re;
  assign                       rx_re = rx_ir & ~rx_fe;

  output        [irqRx_w-1:0]  irqRx;
  output        [irqTx_w-1:0]  irqTx;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire           [run1_w-1:0]  run1_q;
  wire           [run1_w-1:0]  run1_d;
  wire                         run1_weint;
  assign                       run1_weint = 0;

  wire          [rxDat_w-1:0]  rxDat_q;
  wire          [rxDat_w-1:0]  rxDat_d;
  wire                         rxDat_weint;

  wire           [rxSt_w-1:0]  rxSt_q;
  wire           [rxSt_w-1:0]  rxSt_d;
  wire                         rxSt_weint;

  wire          [txDat_w-1:0]  txDat_q;
  wire          [txDat_w-1:0]  txDat_d;
  wire                         txDat_weint;
  assign                       txDat_weint = 0;

  wire           [txSt_w-1:0]  txSt_q;
  wire           [txSt_w-1:0]  txSt_d;
  wire                         txSt_weint;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg run1(run1_q, regBus, run1_d, run1_weint, regWe, regRe, clk, reset);
  defparam run1.w    = run1_w;
  defparam run1.rval = run1_r;
  defparam run1.sgn  = run1_s;
  defparam run1.adr  = R_run1;

  ereg rxDat(rxDat_q, regBus, rxDat_d, rxDat_weint, regWe, regRe, clk, reset);
  defparam rxDat.w    = rxDat_w;
  defparam rxDat.rval = rxDat_r;
  defparam rxDat.sgn  = rxDat_s;
  defparam rxDat.adr  = R_rxDat;

  ereg rxSt(rxSt_q, regBus, rxSt_d, rxSt_weint, regWe, regRe, clk, reset);
  defparam rxSt.w    = rxSt_w;
  defparam rxSt.rval = rxSt_r;
  defparam rxSt.sgn  = rxSt_s;
  defparam rxSt.adr  = R_rxSt;

  ereg txDat(txDat_q, regBus, txDat_d, txDat_weint, regWe, regRe, clk, reset);
  defparam txDat.w    = txDat_w;
  defparam txDat.rval = txDat_r;
  defparam txDat.sgn  = txDat_s;
  defparam txDat.adr  = R_txDat;

  ereg txSt(txSt_q, regBus, txSt_d, txSt_weint, regWe, regRe, clk, reset);
  defparam txSt.w    = txSt_w;
  defparam txSt.rval = txSt_r;
  defparam txSt.sgn  = txSt_s;
  defparam txSt.adr  = R_txSt;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // ---------------------------------------------------------------------------
  // Define wires and registers
  reg               [1:0] rxState;          // 3 possible RX states
  reg               [1:0] txState;          // 4 possible TX states
  
  // RX
  assign rxDat_d      = rx;                 // RX data to RX reg
  assign rxDat_weint  = rx_re;              // write when input is read
  assign rx_ir        = rxState==0;         // state=0/2 -> normal, control
  assign rxSt_weint   = 1;
  assign rxSt_d       = (rxState==1);       // 0/1 -> data is / is not available
  assign irqRx        = (rxState==1);

  // TX
  assign tx           = txDat_q;            // TX reg to TX data
  assign tx_or        = (txState==2);       // write directly after register
  assign txSt_weint   = 1;                  //   output is available
  assign txSt_d       = (txState==0);       // 0/1 TX is / is not busy
  assign irqTx        = (txState==0);

  // ---------------------------------------------------------------------------
  // Main FSM
  always @(posedge clk) begin
    
    // Reset
    if (run1_q==0) begin
      rxState   <= 3;   // Extra reset state
      txState   <= 3;
    end else begin
    
      // -----------------------------------------------------------------------
      // Receive FSM
      // rxState==0 -> wait for data from input
      if (rxState==0) begin
        if (rx_re) begin
          rxState <= 1;
        end
      end else
      
      // rxState==1 -> write wait for data to be read from register
      if (rxState==1) begin
        if (regBus==R_rxDat & regRe) begin
          rxState <= 0;
        end
      end else

      // rxState==3 -> goto state 0
      if (rxState==3) begin
        rxState <= 0;
      end

      // -----------------------------------------------------------------------
      // Transmit FSM
      // txState==0 -> wait for input data
      if (txState==0) begin
        if (regBus==R_txDat & regWe)
          txState <= 1;
      end else
      
      // txState==1 -> wait for data
      if (txState==1) begin
        txState <= 2;
      end else
      
      // txState==2 -> write data to output
      if (txState==2 & tx_we) begin
        txState <= 0;
      end else
      
      // rxState==3 -> goto state 0
      if (txState==3) begin
        txState <= 0;
      end
      
    end // if (run1) ...      
  end //always

endmodule
//=========================================================================================
