//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   extIf
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module extIf(tx, tx_or, tx_ff, tx_we, 
             rx, rx_ir, rx_fe, rx_re, 
             resetOut, 
             irqRx, 
             irqTx, 
             regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter tx_w              = 0;
  parameter rx_w              = 0;

  parameter resetOut_w        = 0;
  parameter irqRx_w           = 0;
  parameter irqTx_w           = 0;

  parameter rxDat_w           = 0;
  parameter rxDat_r           = 0;
  parameter rxDat_s           = 0;
  parameter R_rxDat           = 0;
  parameter rxSt_w            = 0;
  parameter rxSt_r            = 0;
  parameter rxSt_s            = 0;
  parameter R_rxSt            = 0;
  parameter txDat_w           = 0;
  parameter txDat_r           = 0;
  parameter txDat_s           = 0;
  parameter R_txDat           = 0;
  parameter txSt_w            = 0;
  parameter txSt_r            = 0;
  parameter txSt_s            = 0;
  parameter R_txSt            = 0;
  parameter ctrlOn_w          = 0;
  parameter ctrlOn_r          = 0;
  parameter ctrlOn_s          = 0;
  parameter R_ctrlOn          = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output           [tx_w-1:0]  tx;
  output                       tx_or;
  input                        tx_ff;
  output                       tx_we;
  assign                       tx_we = tx_or & ~tx_ff;

  input            [rx_w-1:0]  rx;
  output                       rx_ir;
  input                        rx_fe;
  output                       rx_re;
  assign                       rx_re = rx_ir & ~rx_fe;

  output     [resetOut_w-1:0]  resetOut;
  output        [irqRx_w-1:0]  irqRx;
  output        [irqTx_w-1:0]  irqTx;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire          [rxDat_w-1:0]  rxDat_q;
  wire          [rxDat_w-1:0]  rxDat_d;
  wire                         rxDat_weint;

  wire           [rxSt_w-1:0]  rxSt_q;
  wire           [rxSt_w-1:0]  rxSt_d;
  wire                         rxSt_weint;

  wire          [txDat_w-1:0]  txDat_q;
  wire          [txDat_w-1:0]  txDat_d;
  wire                         txDat_weint;
  assign                       txDat_weint = 0;

  wire           [txSt_w-1:0]  txSt_q;
  wire           [txSt_w-1:0]  txSt_d;
  wire                         txSt_weint;

  wire         [ctrlOn_w-1:0]  ctrlOn_q;
  wire         [ctrlOn_w-1:0]  ctrlOn_d;
  wire                         ctrlOn_weint;
  assign                       ctrlOn_weint = 0;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // Wires for register control
  wire regWeInt;
  wire regReInt;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg rxDat(rxDat_q, regBus, rxDat_d, rxDat_weint, regWe, regRe, clk, reset);
  defparam rxDat.w    = rxDat_w;
  defparam rxDat.rval = rxDat_r;
  defparam rxDat.sgn  = rxDat_s;
  defparam rxDat.adr  = R_rxDat;

  ereg rxSt(rxSt_q, regBus, rxSt_d, rxSt_weint, regWe, regRe, clk, reset);
  defparam rxSt.w    = rxSt_w;
  defparam rxSt.rval = rxSt_r;
  defparam rxSt.sgn  = rxSt_s;
  defparam rxSt.adr  = R_rxSt;

  ereg txDat(txDat_q, regBus, txDat_d, txDat_weint, regWe, regRe, clk, reset);
  defparam txDat.w    = txDat_w;
  defparam txDat.rval = txDat_r;
  defparam txDat.sgn  = txDat_s;
  defparam txDat.adr  = R_txDat;

  ereg txSt(txSt_q, regBus, txSt_d, txSt_weint, regWe, regRe, clk, reset);
  defparam txSt.w    = txSt_w;
  defparam txSt.rval = txSt_r;
  defparam txSt.sgn  = txSt_s;
  defparam txSt.adr  = R_txSt;

  ereg ctrlOn(ctrlOn_q, regBus, ctrlOn_d, ctrlOn_weint, regWe, regRe, clk, reset);
  defparam ctrlOn.w    = ctrlOn_w;
  defparam ctrlOn.rval = ctrlOn_r;
  defparam ctrlOn.sgn  = ctrlOn_s;
  defparam ctrlOn.adr  = R_ctrlOn;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | regWeInt;
  assign regReOut = 0 | regReInt;

// ----------------------------------------------------------------------------------------
//=========================================================================================

//-----------------------------------------------------------------------------------------
  // Define wires and registers
  reg               [2:0] rxState;          // 5 possible RX states
  reg               [1:0] txState;          // 4 possible TX states
  reg              [21:0] cntRx;            // count for waiting a moment after RX
  
  // Manual assignment of regWe/Re
  assign regWeInt = rxState==2 & rx_re;     // write to bus in control mode
  assign regReInt = 0;

  // RX
  assign rxDat_d      = rx;                         // RX data to RX reg
  assign rxDat_weint  = rx_re;                      // write when input is read
  assign rx_ir        = (rxState==0 | rxState==2);  // state=0/2 -> normal, control
  assign rxSt_weint   = 1;
  assign rxSt_d       = (rxState==1);               // 0/1 -> data is / is not available
  assign irqRx        = (rxState==1);

  // TX
  assign tx           = txDat_q;              // TX reg to TX data
  assign tx_or        = (txState==2);         // write directly after register
  assign txSt_weint   = 1;                    //   output is available
  assign txSt_d       = (txState==0);         // 0/1 TX is / is not busy
  assign irqTx        = (txState==0);

  // Access to register bus
  assign regBus = ((rxState==2 & rx_re) | rxState==3) ? rxDat_q : 16'bz;

  // Assign reset out
  assign resetOut = (rxState==5);

//-----------------------------------------------------------------------------------------
  // Main FSM
  always @(posedge clk) begin
    
    // Reset
    if (reset) begin
      rxState   <= 7;   // Extra reset state
      txState   <= 0;
    end else begin
    
      // -----------------------------------------------------------------------
      // Receive FSM
      // rxState==0 -> wait for data from input
      if (rxState==0) begin
        cntRx <= -1;
        if (rx_re)
          if (~ctrlOn_q)
            rxState <= 1;
          else
            rxState <= 2;
      end else
      
      // rxState==1 -> write wait for data to be read from register
      if (rxState==1) begin
        if (regBus==R_rxDat & regRe)
          rxState <= 0;
      end else

      // rxState==2 -> wait for second word to be written to buffer
      if (rxState==2) begin
        cntRx <= cntRx-1;
        if (rx_re)
          rxState <= 3;
        else if (cntRx==0)
          rxState <= 5;
      end else
      
      // rxState==3 -> write to regBus
      if (rxState==3) begin
        rxState <= 0;
      end else

      // rxState==4 -> Not used anymore
      
      // rxState==5 -> virtual state, enable internal reset

      // rxState==7 -> Reset state, prevent RX input, then goto 0
      if (rxState==7) begin
        rxState <= 0;
      end
    
      // -----------------------------------------------------------------------
      // Transmit FSM
      // txState==0 -> wait for input data
      if (txState==0) begin
        if (regBus==R_txDat & regWe)
          txState <= 1;
      end else
      
      // txState==1 -> wait for data
      if (txState==1) begin
        txState <= 2;
      end else
      
      // txState==2 -> write data to output
      if (txState==2 & tx_we) begin
        txState <= 0;
      end
      
    end // if (reset) ...      
  end //always

endmodule
//=========================================================================================
