//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   crc32x8
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module crc32x8(y, y_or, y_ff, y_we, 
               x, x_ir, x_fe, x_re, 
               run1, 
               regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter y_w               = 0;
  parameter x_w               = 0;

  parameter run1_w            = 0;

  parameter numBytes_w        = 0;
  parameter numBytes_r        = 0;
  parameter numBytes_s        = 0;
  parameter R_numBytes        = 0;
  parameter crcRes_w          = 0;
  parameter crcRes_r          = 0;
  parameter crcRes_s          = 0;
  parameter R_crcRes          = 0;

  parameter w                 = 0;
  parameter p0                = 0;
  parameter p1                = 0;
  parameter p2                = 0;
  parameter p3                = 0;
  parameter p4                = 0;
  parameter p5                = 0;
  parameter p6                = 0;
  parameter p7                = 0;

  parameter s_w               = 0;
  parameter s_n               = 0;
  parameter s_m               = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output            [y_w-1:0]  y;
  output                       y_or;
  input                        y_ff;
  output                       y_we;
  assign                       y_we = y_or & ~y_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire       [numBytes_w-1:0]  numBytes_q;
  wire       [numBytes_w-1:0]  numBytes_d;
  wire                         numBytes_weint;

  wire         [crcRes_w-1:0]  crcRes_q;
  wire         [crcRes_w-1:0]  crcRes_d;
  wire                         crcRes_weint;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg numBytes(numBytes_q, regBus, numBytes_d, numBytes_weint, regWe, regRe, clk, reset);
  defparam numBytes.w    = numBytes_w;
  defparam numBytes.rval = numBytes_r;
  defparam numBytes.sgn  = numBytes_s;
  defparam numBytes.adr  = R_numBytes;

  ereg crcRes(crcRes_q, regBus, crcRes_d, crcRes_weint, regWe, regRe, clk, reset);
  defparam crcRes.w    = crcRes_w;
  defparam crcRes.rval = crcRes_r;
  defparam crcRes.sgn  = crcRes_s;
  defparam crcRes.adr  = R_crcRes;

  // --------------------------------------------------------------------------------------
  // Internal status registers
  reg               [s_w-1:0]  s;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Register definitions
  reg               x_reD;
  wire    [w-1:0]   sx;
  reg     [w-1:0]   yR;
  
  // --------------------------------------------------------------------------------------
  // Assign input and output enable
  assign x_ir = run1==1;
  assign y_or = x_reD;
  assign y    = yR;
  
  // Xor input x with bit reversed upper bits of CRC state
  assign sx = x ^ {s[24], s[25], s[26], s[27], s[28], s[29], s[30], s[31]};

  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin
    
    // Reset
    if (run1==0) begin
      s     <= 32'b11111111111111111111111111111111;
      x_reD <= 0;
      
    // Normal operation
    end else begin
      
      // Delayed input read
      x_reD <= x_re;
      
      // Update internal state
      if (x_re) begin

        // Normal CRC operation
        if (|(numBytes_q>>2)) begin
          s <= (s << 8) 
               ^ (sx[0] ? p0 : 0)
               ^ (sx[1] ? p1 : 0)
               ^ (sx[2] ? p2 : 0)
               ^ (sx[3] ? p3 : 0)
               ^ (sx[4] ? p4 : 0)
               ^ (sx[5] ? p5 : 0)
               ^ (sx[6] ? p6 : 0)
               ^ (sx[7] ? p7 : 0);
          yR <= x;
        
        // Give out inverted CRC state and output
        end else begin
          s  <= (s << 8);
          yR <= ~sx;
        
        end // if (numBits_q > 4) ...
      end // if (x_re) ...
    end // if (run1==0) ... else
  end // always

  // Count down the number of bytes
  assign numBytes_d     = numBytes_q-1;
  assign numBytes_weint = x_re;

  // Assign input of CRC register
  assign crcRes_weint = run1==0 | (~|(numBytes_q>>2) & x_re);
  assign crcRes_d     = run1==0 ? 0 : (|(~sx) | crcRes_q);

  // Output
  assign y    = yR;
  assign y_or = x_reD;

endmodule
//=========================================================================================

