//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   cordicRot
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module cordicRot(y, y_or, y_ff, y_we, 
                 x, x_ir, x_fe, x_re, 
                 irqRxDatRd, 
                 runOut, 
                 numOut, 
                 fftWrap, 
                 regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter y_w               = 0;
  parameter x_w               = 0;

  parameter irqRxDatRd_w      = 0;
  parameter runOut_w          = 0;
  parameter numOut_w          = 0;
  parameter fftWrap_w         = 0;

  parameter ph_w              = 0;
  parameter ph_r              = 0;
  parameter ph_s              = 0;
  parameter R_ph              = 0;
  parameter dph_w             = 0;
  parameter dph_r             = 0;
  parameter dph_s             = 0;
  parameter R_dph             = 0;

  parameter wx                = 0;
  parameter wy                = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output            [y_w-1:0]  y;
  output                       y_or;
  input                        y_ff;
  output                       y_we;
  assign                       y_we = y_or & ~y_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  output   [irqRxDatRd_w-1:0]  irqRxDatRd;
  input        [runOut_w-1:0]  runOut;
  input        [numOut_w-1:0]  numOut;
  input       [fftWrap_w-1:0]  fftWrap;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire             [ph_w-1:0]  ph_q;
  wire             [ph_w-1:0]  ph_d;
  wire                         ph_weint;

  wire            [dph_w-1:0]  dph_q;
  wire            [dph_w-1:0]  dph_d;
  wire                         dph_weint;
  assign                       dph_weint = 0;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg ph(ph_q, regBus, ph_d, ph_weint, regWe, regRe, clk, reset);
  defparam ph.w    = ph_w;
  defparam ph.rval = ph_r;
  defparam ph.sgn  = ph_s;
  defparam ph.adr  = R_ph;

  ereg dph(dph_q, regBus, dph_d, dph_weint, regWe, regRe, clk, reset);
  defparam dph.w    = dph_w;
  defparam dph.rval = dph_r;
  defparam dph.sgn  = dph_s;
  defparam dph.adr  = R_dph;

  // --------------------------------------------------------------------------------------
  // Included instances
  `include "cordicRotCore_0x12.v"
  `include "cordicRotCore_1x12.v"
  `include "cordicRotCore_2x12.v"
  `include "cordicRotCore_3x12.v"
  `include "cordicRotCore_4x12.v"
  `include "cordicRotCore_5x12.v"
  `include "cordicRotCore_6x12.v"
  `include "cordicRotCore_7x12.v"
  `include "cordicRotCore_8x12.v"
  `include "cordicRotCore_9x12.v"
  `include "buf2_20.v"

  // --------------------------------------------------------------------------------------
  // Define connections
  assign y                         = buf2_20_y;
  assign y_or                      = buf2_20_y_or;
  assign buf2_20_y_ff              = y_ff;


  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | cordicRotCore_0x12_regWe | cordicRotCore_1x12_regWe | cordicRotCore_2x12_regWe
                      | cordicRotCore_3x12_regWe | cordicRotCore_4x12_regWe | cordicRotCore_5x12_regWe
                      | cordicRotCore_6x12_regWe | cordicRotCore_7x12_regWe | cordicRotCore_8x12_regWe
                      | cordicRotCore_9x12_regWe | buf2_20_regWe;
  assign regReOut = 0 | cordicRotCore_0x12_regRe | cordicRotCore_1x12_regRe | cordicRotCore_2x12_regRe
                      | cordicRotCore_3x12_regRe | cordicRotCore_4x12_regRe | cordicRotCore_5x12_regRe
                      | cordicRotCore_6x12_regRe | cordicRotCore_7x12_regRe | cordicRotCore_8x12_regRe
                      | cordicRotCore_9x12_regRe | buf2_20_regRe;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // Define some parameters from included files
  parameter wp = cordicRotCore_0x12_wp;

  // --------------------------------------------------------------------------------------
  // Register definitions
  wire    [wx+1:0]  xRe,  xIm;
  wire    [wy-1:0]  yRe,  yIm;
  reg     [wx+1:0]  xRe1, xIm1, xRe4, xIm4, xRe7, xIm7;
  reg     [wp-1:0]  ph1, ph4, ph7;
  reg     [wy-1:0]  yReD, yImD;
  reg        [2:0]  x_reD;
  reg        [6:0]  cnt;
  reg        [7:0]  cntOut;
  reg               tgl;
  reg        [6:0]  fftWrapP1;

  reg   [ph_w-1:0]  ph_nxt, phAdd, phSub;
  
  wire              startOutFsm, runOutEq0, cntOutEq0;
  reg               runOutEq0D, cntOutEq0D;
  
  // --------------------------------------------------------------------------------------
  // Assign input and output enable (delayed by pipeline)
  assign x_ir = buf2_20_x_ir & runOut!=0;
    
  // Start system if runOut switches from 0 -> 1 or 2
  assign runOutEq0    = runOut==0;
  assign startOutFsm  = runOutEq0D==1 & runOutEq0==0;

  // Assign IRQ output
  assign cntOutEq0  = cntOut==0;
  assign irqRxDatRd = cntOutEq0D==0 & cntOutEq0;

  // --------------------------------------------------------------------------------------
  // Calculate next phase
  always @(*) begin
    
    // Use this for FFT wrap comparison
    fftWrapP1 = {3'b000, fftWrap}+1;

    // modePh==1, Linear update with possible fftWrap
    if (runOut==1) begin
      phAdd  = dph_q;
      if (fftWrapP1==cnt) begin
        phSub = dph_q<<6;
      end else begin
        phSub = 0;
      end
    end else
    
    // modePh==2, jump back and forth
    begin
      if (tgl==0) begin
        phAdd = dph_q<<6;
        phSub = 0;
      end else begin
        phAdd = dph_q;
        if (fftWrapP1==cnt) begin
          phSub = dph_q<<7;
        end else begin
          phSub = dph_q<<6;
        end
      end
    end
  
    ph_nxt = ph_q + phAdd - phSub;
  
  end // always @(*)
  
  // Update external register
  assign ph_d      = ph_nxt;
  assign ph_weint  = x_re;

  // --------------------------------------------------------------------------------------
  // Signal processing behaviour  
  assign xRe = {{2{x[  wx-1]}}, x[  wx-1: 0]};
  assign xIm = {{2{x[2*wx-1]}}, x[2*wx-1:wx]};
  
  assign cordicRotCore_0x12_xReIn = xRe;
  assign cordicRotCore_0x12_xImIn = xIm;
  assign cordicRotCore_0x12_phIn  = ph_q[ph_w-1:6];

  assign cordicRotCore_1x12_xReIn = cordicRotCore_0x12_xReOut;
  assign cordicRotCore_1x12_xImIn = cordicRotCore_0x12_xImOut;
  assign cordicRotCore_1x12_phIn  = cordicRotCore_0x12_phOut;

  assign cordicRotCore_2x12_xReIn = xRe1;
  assign cordicRotCore_2x12_xImIn = xIm1;
  assign cordicRotCore_2x12_phIn  = ph1;

  assign cordicRotCore_3x12_xReIn = cordicRotCore_2x12_xReOut;
  assign cordicRotCore_3x12_xImIn = cordicRotCore_2x12_xImOut;
  assign cordicRotCore_3x12_phIn  = cordicRotCore_2x12_phOut;

  assign cordicRotCore_4x12_xReIn = cordicRotCore_3x12_xReOut;
  assign cordicRotCore_4x12_xImIn = cordicRotCore_3x12_xImOut;
  assign cordicRotCore_4x12_phIn  = cordicRotCore_3x12_phOut;

  assign cordicRotCore_5x12_xReIn = xRe4;
  assign cordicRotCore_5x12_xImIn = xIm4;
  assign cordicRotCore_5x12_phIn  = ph4;

  assign cordicRotCore_6x12_xReIn = cordicRotCore_5x12_xReOut;
  assign cordicRotCore_6x12_xImIn = cordicRotCore_5x12_xImOut;
  assign cordicRotCore_6x12_phIn  = cordicRotCore_5x12_phOut;

  assign cordicRotCore_7x12_xReIn = cordicRotCore_6x12_xReOut;
  assign cordicRotCore_7x12_xImIn = cordicRotCore_6x12_xImOut;
  assign cordicRotCore_7x12_phIn  = cordicRotCore_6x12_phOut;

  assign cordicRotCore_8x12_xReIn = xRe7;
  assign cordicRotCore_8x12_xImIn = xIm7;
  assign cordicRotCore_8x12_phIn  = ph7;

  assign cordicRotCore_9x12_xReIn = cordicRotCore_8x12_xReOut;
  assign cordicRotCore_9x12_xImIn = cordicRotCore_8x12_xImOut;
  assign cordicRotCore_9x12_phIn  = cordicRotCore_8x12_phOut;

  // Shift >>1, Symmetric saturation 12->10 bit...
  sat sat_re(yRe, cordicRotCore_9x12_xReOut);
  defparam sat_re.x_w = wx+2;
  defparam sat_re.y_w = wy;
  
  sat sat_im(yIm, cordicRotCore_9x12_xImOut);
  defparam sat_im.x_w = wx+2;
  defparam sat_im.y_w = wy;

  // Connect output FIFO buffer
  assign  buf2_20_run1  = runOut!=0;
  assign  buf2_20_x_fe  = ~x_reD[2];
  assign  buf2_20_x     = {yIm, yRe};

  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin
    
    // -------------------------------------------
    // Delayed runOut==0 + cntOut
    runOutEq0D <= runOutEq0;
    cntOutEq0D <= cntOutEq0;
    
    // Reset
    if (runOut==0) begin
      x_reD  <= 0;
      cntOut <= 0;
      
    // Normal operation
    end else begin
      
      if (startOutFsm) begin
        cntOut  <= runOut==1 ? numOut : numOut<<1;
        cnt     <= numOut;
        tgl     <= 0;
      end else begin
      
        // Output sample counter
        if (y_we) begin
          cntOut <= cntOut-1;
        end
        
        // Run FSM only if output buffer FIFO accepts data
        if (buf2_20_x_ir) begin

          // Set output counter
          if (x_re & cnt!=0 & (tgl==1 | runOut==1)) begin
            cnt <= cnt-1;
          end

          // Set output toggling for mode 2
          if (x_re & runOut==2) begin
            tgl <= ~tgl;
          end      
           
          // Delay input read enable + IRQ
          x_reD <= {x_reD[1:0], x_re};
          
          // Delayed values
          xRe1  <= cordicRotCore_1x12_xReOut;
          xRe4  <= cordicRotCore_4x12_xReOut;
          xRe7  <= cordicRotCore_7x12_xReOut;
                            
          xIm1  <= cordicRotCore_1x12_xImOut;
          xIm4  <= cordicRotCore_4x12_xImOut;
          xIm7  <= cordicRotCore_7x12_xImOut;
          
          ph1   <= cordicRotCore_1x12_phOut;
          ph4   <= cordicRotCore_4x12_phOut;
          ph7   <= cordicRotCore_7x12_phOut;
        
        end // if (buf2_20_x_ir) begin  
      end // if (startOutFsm) ... else
    end // if (runOut_q==0) ... else
  end // always
  
endmodule
//=========================================================================================

