//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16PramDec
//
// PURPOSE:
//   Pram decoder for the X16 processor
//
// INPUT:
//   x16Adr[13:0]:        14 bit address from AGU unit
//   regBus[15:0]:        Register bus 
//   regWe:               Register bus write enable
//
// OUTPUT:
//   pramDat[31:0]:       PRAM data
//   pramDat[m-1:0]:      PRAM data
//   pramWe:              PRAM write enable
//   pramRe:              PRAM read enable
//
// DESCRIPTION:
//   This module provides a multiplexor for initializing the PRAM via the
//   internal register bus. 
//   During normal operation, x16Adr in simply passed to the PRAM for 
//   reading. x16Adr is then directly addressing a 32 bit wide memory
//   element. 
//   When initializing the PRAM via regbus, the address must first be written
//   into and address register. As the bus is only 16 bit wide, the address
//   given refers to the memory as 16 bit wide memory memory, too. The first
//   16 bit are buffered internally, while the 2nd 16 bit will write the PRAM.
//   If the data bytes to be written into PRAM were [0x20, 0x21, 0x22, ...], and
//   the data/address bus address were 3 and 2, respecitvely, the initialization
//   sequence would be
//     RegBus      RegWe
//       2           1       -> use the address register
//       0           0       -> set address register value
//       3           1       -> use the data register
//     0x2120        0       -> set the data register value
//       2           1       -> use the address register
//       1           0       -> set address register value
//       3           1       -> use the data register
//     0x2322        0       -> write all 32 bit (0x2120) into PRAM
//       2           1       -> ...
//       2           0
//       3           1
//     0x2524        0
//       2           1
//       3           0
//       3           1
//     0x2726        0
//       2           1
//       4           0
//       3           1
//     0x2928        0
//       ...
//
// HISTORY:
//   12-Oct-06, Dirk Sommer
//     Intial version
//
//========================================================================================

//========================================================================================
module x16PramDec(pramAdr,
                  pramDat,
                  pramWe,
                  x16Adr,
                  regBus,
                  regWe,
                  clk);

  // -------------------------------------------------------------------------------------
  // Define parameters - to be overwritten externally
  parameter m               = 8;            // Pram address width                 

  parameter adrAdr          = 2;
  parameter adrDat          = 3;

  // Outputs to Dram
  output   [m-1:0]  pramAdr;                // Address output                     
  output    [31:0]  pramDat;                // Data input to pram                 
  output            pramWe;                 // Write enable output                

  // Inputs from Dec
  input     [13:0]  x16Adr;                 // Address input                      

  // General register bus
  input     [15:0]  regBus;
  input             regWe;

  input             clk;                    // Clock                              
  
  // -------------------------------------------------------------------------------------
  // Internal registers
  reg               regAdrWe, regDat0We, regDat1We;
  reg       [ m:0]  regAdr;
  reg       [15:0]  regDat0;

  // -------------------------------------------------------------------------------------
  // Operation
  always @(posedge clk) begin

    // Write enable signals -> they are delayed, because the effective write is in the next cycle
    regAdrWe  <= (regBus==adrAdr) & regWe;                  // Write enable for internal address reg.
    regDat0We <= (regBus==adrDat) & regWe & (regAdr[0]==0); // Write enable for 16 bit buffer (lower 16 bit)
    regDat1We <= (regBus==adrDat) & regWe & (regAdr[0]==1); // Write enable for memory (upper + lower 16 bit)
    
    // Write the internal address or data register
    if (regAdrWe)
      regAdr <= regBus[m:0];                                // Set address from bus, the LSB indicates
                                                            // Buffering or writing to PRAM
    if (regDat0We)
      regDat0 <= regBus;                                    // Set lower 16 bit from bus
    
  end //always

  // -------------------------------------------------------------------------------------
  // Output multiplexing for Pram
  assign pramAdr = regDat1We ? regAdr[m:1] : x16Adr[m-1:0];
  assign pramWe  = regDat1We;
  assign pramDat = {regBus, regDat0};
  
endmodule
//========================================================================================

