//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16Pcu
//
// PURPOSE:
//   Program control unit for X16
//
// INPUT:
//  pdatIn:     Pram input data
//  muxIn:      Input from DEC
//  cmd:        Commands
//  halt:       Halt flag from decoder
//
// OUTPUT:
//  padr:       Pram output address
//  pdatOut:    Pram output data
//  muxOut:     Output to muxer
//
// DESCRIPTION:
//   Output PC[15:2], assuming a 32 bit wide PRAM (24 bit would be sufficient)
//   - Determine the length of the current command (-> pdatLen)
//   - Evaluate the current command input
//     o PcuFetch:
//       Simply fetch the next command, PC = PC+pdatLen
//     o PcuLlc:
//       Load LC (loop counter), use decIn, PC = PC+pdatLen
//     o PcuJmpR:
//       Jump relative, use decIn, PC = PC+decIn
//     o PcuJmpX:
//       After a jump, fetch the data from PRAM as usual, but
//       set PC = PC+4 in order to fill the data buffer properly
//     o PcuJmpA:
//       Jump absolute, use PC = decIn
//     o PcuRet:
//       Jump absolute, return from subfunction, use PC = aguIn
//     o PcuWait:
//       Wait until LC==0, decrease LC = LC-1
//    An internal buffer saves the PRAM data and passes it the the decoder
//    depending on the lower 2 bits of the PC.
//
// HISTORY:
//   11-11-06, Dirk Sommer
//     Initial version
//
//========================================================================================

//========================================================================================
module x16Pcu(padr,
              pdatOut,
              muxOut,
              pdatIn,
              muxIn,
              cmd,
              halt,
              clk, reset);

  // ----------------------------------------------------------------------------------------
  // Define ports
  output              [13:0]  padr;                         // pram output address
  output              [23:0]  pdatOut;                      // pram output data
  output              [15:0]  muxOut;                       // output to muxer
  input               [31:0]  pdatIn;                       // pram input data
  input               [15:0]  muxIn;                        // input from DEC
  input               [ 1:0]  cmd;                          // commands
  input                       halt;                         // halt flag from decoder
  input                       clk, reset;

  // ----------------------------------------------------------------------------------------
  // Use registers for output
  reg                 [23:0]  pdatOut;
  
  // Global X16 definitions
  `include "def_X16.v"

  // ----------------------------------------------------------------------------------------
  // Define a few internal wires and registers
  reg                 [15:0]  PC, PC_nxt, PC_prv;
  reg                 [23:0]  pdat;   
  reg                 [31:0]  pdatBuf;
  reg                 [31:0]  pdatBufIn;   
  reg                 [ 1:0]  pdatLen;
  reg                 [15:0]  muxOut;
  reg                 [15:0]  pcuAddIn1, pcuAddIn2;
  wire                [15:0]  pcuAddOut;
  wire                [63:0]  mb;
  reg                         haltD;
  wire                [31:0]  pdatInUse;
  
  // ----------------------------------------------------------------------------------------
  // Determine PramAdr output
  assign padr   = PC_nxt[15:2];

  // Define internal state of the "FIFO"
  always @(posedge clk)
    if ((PC[2]^PC_nxt[2]) & ~halt) begin
      pdatBuf   <= pdatInUse;
    end

  // ----------------------------------------------------------------------------------------
  // Make sure to buffer the input PRAM data in case of halt
  always @(posedge clk) begin
    haltD <= halt;
    if (~haltD)
      pdatBufIn <= pdatIn;
  end
  assign pdatInUse = haltD ? pdatBufIn : pdatIn;
  
  // ----------------------------------------------------------------------------------------
  // Define PCU adder
  assign pcuAddOut = pcuAddIn1 + pcuAddIn2;

  // ----------------------------------------------------------------------------------------
  // Evaluate command length -> pcAdd
  always @(*) begin

    // Get pdat length - evaluate current command
    if (((pdat[4:0]==5'b00000) & (pdat[7:5]==X16Cal)) |  // call
         (pdat[4:0]==5'b00100) |                         // ldv
         (pdat[4:0]==5'b00001) ) begin                   // gta
      pdatLen = 3;        
    end else
        
    if (((pdat[4:0]==5'b00000) & (pdat[7:5]==X16Llc | 
                                  pdat[7:5]==X16Wtc | 
                                  pdat[7:5]==X16Lop)) |
          pdat[0]==1'b1) begin
      pdatLen = 2;
    end else
    
    begin
      pdatLen = 1;
    end

    // --------------------------------------------------------------------------------------
    // Define 1st adder input
    if (cmd==PcuJmpR | cmd==PcuJmpA)
      pcuAddIn1 = PC_prv;
    else
      pcuAddIn1 = PC;
      
    // Define 2nd adder input
    case (cmd)
      PcuJmpR:  pcuAddIn2 = muxIn;      // Jump relative
      PcuJmpX1: pcuAddIn2 =  4;         // Add 4 to PC after jump
      PcuJmpA:  pcuAddIn2 = -4;         // For call
      PcuFetch: pcuAddIn2 = pdatLen;    // Normal case
    endcase

    // Get next PC - use PCU adder output except for direct load of PC
    if (cmd==PcuJmpA)
      PC_nxt = muxIn;
    else
      PC_nxt = pcuAddOut;               // Simply add (see above)
          
  end
  
  // ----------------------------------------------------------------------------------------
  // Assign output data for decoder
  assign mb = {pdatInUse, pdatBuf};     // Memory and buffer output
  
  always @(*) begin
    case (PC[1:0])                      // Select memory output depending
      2'b00: pdat = mb[23: 0];          // On the PC lower bits
      2'b01: pdat = mb[31: 8];
      2'b10: pdat = mb[39:16];
      2'b11: pdat = mb[47:24];
    endcase
  end

  // ----------------------------------------------------------------------------------------
  // State machine update
  always @(posedge clk) begin
    
    // Synchronous reset of the state machine  
    if (reset) begin
      PC  <= -4;
    end else

    // Normal operation
    if (~halt) begin    
      PC        <= PC_nxt;
      PC_prv    <= PC;                          // save a history of PC for call
      muxOut    <= pcuAddOut;                   // muxOut <= PC_prv-4 for PcuJmpA (i.e. call)
      pdatOut   <= pdat;
    end

  end //always
endmodule
//========================================================================================
    
