//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16DramDec
//
// PURPOSE:
//   Data RAM address decoding and access
//
// INPUT:
//  dramDatIn:      Data input from dram
//  x16Adr:         Address input
//  x16DatIn:       Data input from X16Reg
//  x16We:          Write enable input
//  x16Re:          Read enable input
//  regWeIn:        Register bus write enable input
//  regReIn:        Register bus read enable input
//
// OUTPUT:
//  dramAdr:        Address output                                     
//  dramDatOut:     Data output to dram                                
//  dramWe:         Write enable output                                
//  x16DatOut:      Data output to X16Reg                              
//  x16Wait:        Wait flag                                          
//  regWeOut:       Output write enable to regBus                      
//  regReOut:       Output read enable to regBus                       
//
//
// DESCRIPTION:
//   The DRAM decoder maps write and read signal from the PCU to either the
//   DRAM or the register bus, depending on the address and the "m" parameter.
//   All addresses 0..2^m-1 are mapped to the register bus, all addresses
//   above are mapped to the DRAM in a common address space.
//   Additionally, the memory can be written via 2 special RegBus addresses
//   (adrAdr, adrDat), in order to write the DRAM during boot.
//
// HISTORY:
//   11-Nov-06, Dirk Sommer
//     Initial version
//
//====================================================================================

//========================================================================================
module x16DramDec(dramAdr,
                  dramDatOut,
                  dramDatIn, 
                  dramWe, 
                  x16Adr,  
                  x16DatOut, 
                  x16DatIn, 
                  x16We,  
                  x16Re, 
                  x16Wait,
                  regBus, 
                  regWeOut, 
                  regReOut, 
                  regWeIn, 
                  regReIn,
                  clk);

  // -------------------------------------------------------------------------------------
  // Define parameters - must be overwritten from outside
  parameter m               = 9;    // Dram address width

  parameter adrAdr          = 4;
  parameter adrDat          = 5;

  // Outputs to Dram
  output   [m-1:0]  dramAdr;        // Address output
  output    [15:0]  dramDatOut;     // Data output to dram
  input     [15:0]  dramDatIn;      // Data input from dram
  output            dramWe;         // Write enable output

  wire              dramRe;         // Read enable output

  // Inputs from x16Dec
  input     [15:0]  x16Adr;         // Address input
  output    [15:0]  x16DatOut;      // Data output to X16Reg
  input     [15:0]  x16DatIn;       // Data input from X16Reg
  input             x16We;          // Write enable input
  input             x16Re;          // Read enable input
  output            x16Wait;        // Wait flag

  // General register bus
  inout     [15:0]  regBus;
  output            regWeOut;       // Output write enable to regBus
  output            regReOut;       // Output read enable to regBus
  input             regWeIn;
  input             regReIn;

  input             clk;            // Clock
  
  // -------------------------------------------------------------------------------------
  // More registers/wires
  reg               regAdrWe, regDatWe, regDatRe;
  reg       [15:0]  regAdr;
  reg       [15:0]  regBusOut;
  reg       [15:0]  x16DatIn2;

  wire      [15:0]  pat0, pat1;
  wire              isRegAdr, isDramAdr;                    // for address decoding

  reg               regReOutD, regWeOutD, dramReD;          // delayed version

//========================================================================================
  // Operation
  always @(posedge clk) begin

    // Write enable signals -> delayed, because the effective write is in the next cycle
    regAdrWe <= (regBus == adrAdr) & regWeIn;              // Write enable for internal address reg.
    regDatWe <= (regBus == adrDat) & regWeIn;              // Write enable for mem
    regDatRe <= (regBus == adrDat) & regReIn;              // Read  enable for mem

    // Write the internal address or data register, use regbus one cycle after regWeIn
    if (regAdrWe)
      regAdr <= regBus;                                    // Set address from bus
    
    // Delay some inputs
    x16DatIn2 <= x16DatIn;

    // Delayed signals for regBus
    regReOutD   <= regReOut;
    regWeOutD   <= regWeOut;
    dramReD     <= dramRe;

  end // always

  //-------------------------------------------------------------------------------------
  // Address decoding - check if we access Dram or RegBus
  assign pat0 = 16'd0;
  assign pat1 = 16'd1;
  assign isRegAdr   = x16Adr[15:m]==pat0[15-m:0];           // 0..2^m-1 are registers
  assign isDramAdr  = x16Adr[15:m]==pat1[15-m:0];           // 2^m..2^(m+1)-1 is Dram
                                                            // -> might use dramAdr??
  //-------------------------------------------------------------------------------------
  // Output multiplexing for Dram
  assign dramAdr    = (x16We|x16Re) ? x16Adr[m-1:0] : regAdr[m-1:0];// use address from AGU or regAdr 
  assign dramWe     =  regDatWe | (x16We & isDramAdr);              // "or" internal and external
  assign dramRe     = (regDatRe | (x16Re & isDramAdr)) & ~dramReD;  // write and read signal
  assign dramDatOut = x16We ? x16DatIn : regBus;                    // x16 is prioritized

  //-------------------------------------------------------------------------------------
  // Output multiplexing for regBus
  assign regWeOut  = (x16We & isRegAdr) & ~regWeOutD;
  assign regReOut  = (x16Re & isRegAdr) & ~regReOutD;
  assign x16Wait   = (regReOut | regWeOut | dramRe);        // if read/write regBus -> wait a cycle
  
  always @(*)
    case ({regWeOutD, regWeOut, regReOut, regDatRe})
      4'b0001 : regBusOut = dramDatIn;                      // read DRAM data
      4'b0010 : regBusOut = x16Adr;                         // read a register
      4'b0100 : regBusOut = x16Adr;                         // write a register (adr)
      4'b1000 : regBusOut = x16DatIn2;                      // write a register (dat)
      default : regBusOut = 16'bz;
    endcase
  
  assign regBus = regBusOut;

  // -------------------------------------------------------------------------------------
  // Output multiplexing for x16
  assign x16DatOut = regReOutD ? regBus : dramDatIn;
  
endmodule
//========================================================================================

