//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   spRam1
//
// PURPOSE:
//   Single ported RAM
//
// INPUT:
//   d:       Data input
//   adr:     Address
//   we:      Write enable
//
// OUTPUT:
//   q:       Data output
//
// PARAMETER:
//   m:       Address bit width
//   n:       Number of elements
//   w:       Data bit width
//
// DESCRIPTION:
//   Single ported memory implementation. The output is read with 1 clock
//   cycle delay with respect to the adr setting. Data is present in
//   memory with 2 cycles delay.
//
// HISTORY:
//   01-Jan-06, Dirk Sommer
//     Initial version
//
//==============================================================================

//==============================================================================
module spRam1(q, d, adr, we, clk);

  // define parameters
  parameter m = 10;                     // Address width
  parameter n = 1024;                   // Number of elements
  parameter w = 8;                      // Data width
  parameter fpgaDef = "none";           // FPGA define, placeholder here

  // define inputs and outputs
  output    [w-1:0]     q;              // Data output
  input     [w-1:0]     d;              // Data input
  input     [m-1:0]     adr;            // Read/write address
  input                 we;             // Write enable
  input                 clk;            // Clock
  
  // define internal states
  reg       [w-1:0]     mem[n-1:0];     // Memory
  reg                   weD;            // Delayed read enable
  reg       [m-1:0]     adrD;           // Delayed address
  reg       [w-1:0]     dD;             // Delayed data
  
  // synchronous memory operation - input is registered (as in Altera Megafunction)
  always @(posedge clk) begin
    weD     <= we;
    adrD    <= adr;
    dD      <= d;
    if (weD)
      mem[adrD] <= dD;
  end

  // assign output
  assign q = mem[adrD];                 // Output if read enable in
                                        // the previous cycle
endmodule
//==============================================================================

