//=========================================================================================
//  Copyright (C) Dirk Sommer, 2005
//=========================================================================================
//
//  NAME:
//    sat
//
//  PURPOSE:
//    Saturation of an integer number
//
// PARAMETERS:
//    x_w:    Width of inputs x 
//    y_w:    Width of output y
//
//  INPUTS:
//    x:      Saturation input
//  
//  OUTPUTS:
//    y:      Saturation output
//
//  DESCRIPTION:
//    Saturate a signed integer number as
//
//          { -(2^(y_w-1)-1)    for  x < -(2^(y_w-1)-1)
//      y = {  x                for  
//          {  (2^(y_w-1)-1)    for  x >  (2^(y_w-1)-1)
//
//    Note that operation of this module only makes sense if y_w < x_w
//
//  HISTORY:
//    09-Feb-2005, Sommer:
//      Definition of the module
//
//=========================================================================================

//=========================================================================================
module sat(y, x);

//module parameters
  parameter x_w = 17;
  parameter y_w = 16;

//ports
  output    [y_w-1:0]   y;
  input     [x_w-1:0]   x;
  
//define some wires for the ease of operation
  wire                  op;                                 //positive overflow
  wire                  on;                                 //negative overflow
  reg       [y_w-1:0]   y;
  
//-----------------------------------------------------------------------------------------
//do the saturation depending on the sign of x
  assign op = (x[x_w-1]==0) &  ( |x[x_w-2:y_w-1]);
  assign on = (x[x_w-1]==1) & ((~&x[x_w-2:y_w-1]) | (~|x[y_w-2:0]));
  
  always @(*)
    case ({on, op})
      2'b01     : y = {1'b0, {y_w-1{1'b1}}};                // 2^(y_w-1)-1
      2'b10     : y = {1'b1, {y_w-2{1'b0}}, 1'b1};          //-2^(y_w-1)-1
      default   : y = x[y_w-1:0];                           //no saturation
    endcase  

endmodule
//=========================================================================================
