//=========================================================================================
//  Copyright (C) BAY9, 2011
//=========================================================================================
//
//  NAME:
//    fifoB
//
//  PURPOSE:
//    FIFO
//
//  INPUT:
//    d:    Data input 
//    re:   Read enable
//    we:   Write enable
//    ird:  Input ready
//    ord:  Output ready
//
//  OUTPUT:
//    w:    Data output
//    fe:   FIFO empty
//    ff:   FIFO full
//    ff2:  FIFO full, but only when re=0
//    fuf:  FIFO underflow
//    fof:  FIFO overflow
//
//  PARAMETER:
//    m  = Address bit width
//    n  = Number of elements
//    w  = Data bit width
//    a  = FIFO address (only needed to short circuit the FIFO if set -1)
//    
//  DESCRIPTION:
//    Provide FIFO operation. Data is written to the FIFO at the rising clk edge when
//    we=1, and read from the output if re=1. The FIFO can be bypassed if the "address"
//    value is set to -1 (remark, the value is normally only needed for the version of
//    fifoA, which can be reset via regBus by writing the FIFO address). Fifo underflow
//    and overflow flags are provided, these flags are sticky and return to 0 only
//    upon reset.
//
//  HISTORY:
//    29-Jan-2011, Dirk Sommer
//      Initial version, copied from FIFO-A

// ========================================================================================
module fifoB(q, fe, ff, ff2, fuf, fof, d, re, we, ird, ord, rst, clk);

  // Define parameters - typically overwritten from outside
  parameter m = 10;                     // Address width 
  parameter n = 1024;                   // Number of elements
  parameter w = 16;                     // Data width
  parameter a = 0;                      // FIFO address

  // Define inputs and outputs
  output [w-1:0] q;                     // Data output
  output         fe;                    // Fifo empty
  output         ff;                    // Fifo full
  output         ff2;                   // Fifo full, but only when re=0
  output         fuf;                   // Fifo underflow
  output         fof;                   // Fifo overflow

  input  [w-1:0] d;                     // Data input
  input          re;                    // Read enable
  input          we;                    // Write enable
  input          ird;                   // Input ready
  input          ord;                   // Output ready
  input          rst;                    // FIFO reset
  input          clk;                   // Clock
  
  // Define internal states
  reg [w-1:0]   mem[n-1:0];             // Fifo memory
  reg [m:0]     cnt;                    // Number of fill level of the fifo    
  reg [m-1:0]   radr;                   // Read address
  reg [m-1:0]   wadr;                   // Write address
  reg           fe1;                    // Fifo empty
  reg           ff1;                    // Fifo full
  reg           fuf;                    // Fifo underflow
  reg           fof;                    // Fifo overflow

  // Main FSM
  always @(posedge clk)
  
    // Reset
    if (rst) begin                      // Synchronous reset
      cnt     <= 0;                     // Number of words in the fifo
      radr    <= 0;                     // Read address
      wadr    <= 0;                     // Write address
      fe1     <= 1;       
      ff1     <= 0;
      fuf     <= 0;
      fof     <= 0;
      
    // Normal operation
    end else begin                  
      fe1 <= (cnt==0 & !we) | (cnt==1     & re & !we);
      ff1 <= (cnt==n & !re) | (cnt==(n-1) & we & !re);
      fuf <= fuf | (cnt==0 & re);                       // underflow and overflow
      fof <= fof | (cnt==n & we & !re);                 // are sticky
      cnt <= cnt+we-re;                                   
      if (re) begin
        if (radr != (n-1))
          radr <= radr + re;
        else
          radr <= 0;
      end    
      if (we) begin
        mem[wadr] <= d;
        if (wadr != (n-1))
          wadr <= wadr + we;
        else
          wadr <= 0;
      end          
    end // Normal operation

  // Assign outputs
  assign ff2    = ff1 & !re;  // 2nd fifo full flag, fifo is not really full if re!

  // ------------------------------------------------------------------------------
  // Check here if the FIFO size is 0 -> adr=-1 -> connect the modules directly
  assign q  = (a!=-1) ? mem[radr] : d;         // Always read
  assign fe = (a!=-1) ? fe1       : ~ord;  
  assign ff = (a!=-1) ? ff1       : ~ird;

endmodule
//========================================================================================
        
