//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   dpRam1
//
// PURPOSE:
//   Dual ported RAM
//
// INPUT:
//   d:       Data input
//   rdAdr:   Read address
//   wrAdr:   Write address
//   we:      Write enable
//
// OUTPUT:
//   q:       Data output
//
// PARAMETER:
//   m:       Address bit width
//   n:       Number of elements
//   w:       Data bit width
//
// DESCRIPTION:
//   Dual ported memory implementation. The output is read with 1 clock
//   cycle delay with respect to the rdAdr setting. Data is present in
//   memory with 2 cycles delay.
//
// HISTORY:
//   01-Jan-06, Dirk Sommer
//     Initial version
//
//==============================================================================

//==============================================================================
module dpRam1(q, d, rdAdr, wrAdr, we, clk);

  // Define parameters
  parameter m = 10;                     // Address width
  parameter n = 1024;                   // Number of elements
  parameter w = 8;                      // Data width
  parameter fpgaDef = "none";           // FPGA define, placeholder here

  // Define inputs and outputs
  output    [w-1:0]     q;              // Data output
  input     [w-1:0]     d;              // Data input
  input     [m-1:0]     rdAdr;          // Read address
  input     [m-1:0]     wrAdr;          // Write address
  input                 we;             // Write enable
  input                 clk;            // Clock
  
  // Define internal states
  reg       [w-1:0]     dpm[n-1:0];     // Memory
  reg                   weD;            // Delayed read enable
  reg       [m-1:0]     rdAdrD;         // Delayed address
  reg       [m-1:0]     wrAdrD;         // Delayed address
  reg       [w-1:0]     dD;             // Delayed data
  
  // Synchronous memory operation - input is registered (as in Altera Megafunction)
  always @(posedge clk) begin
    weD     <= we;
    rdAdrD  <= rdAdr;
    wrAdrD  <= wrAdr;
    dD      <= d;
    if (weD)
      dpm[wrAdrD] <= dD;
  end

  // Assign output
  assign q = dpm[rdAdrD];               // Output if read enable in
                                        // the previous cycle
endmodule
//==============================================================================
        
