//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   bridge
//
// PURPOSE:
//   Register bus bridge
//
// CONTROL OUTPUT:
//   irqMst:   Master IRQ
//   irqSlv:   Slave IRQ
//
// EREG:
//   Master:
//     fifoMst:  Write or read message to / from FIFO
//     ctrlMst:  Access control register
//   Slave:
//     fifoSlv:  Data to transmit
//     ctrlSlv:  Access to control register
//
// DESCRIPTION:
//   Provide a bridge between 2 domains of the registers bus. The module has
//   different IO ports for the master and the slave register bus. It contains
//   2 FIFOs of 16 entries each for the 2 directions in order to allow sending
//   messages. The FIFOs can be written and read via the register bus. The
//   addresses write/read the FIFO as follows:
//
//     From master bus:
//       R_fifoMst, write: write Mst2Slv-FIFO
//       R_fifoMst, read:  read  Slv2Mst-FIFO
//
//     From slave bus:
//       R_fifoSlv, write: write Slv2Mst-FIFO
//       R_fifoSlv, read:  read  Mst2Slv-FIFO
//
//   An interrupt is issued at the receiver side when data is written to a
//   FIFO.  The IRQ keeps pending as long as the corresponding FIFO is not empty
//   and is cleared automatically when empty again. The FIFO status can also be
//   read from the control/status which contains the following entries:
//
//     Bit | Name   | R/W | Purpose
//     ----+--------+-----+--------------------------------
//      0  | M2SFe  |  R  | Master to slave FIFO empty flag
//      1  | M2SFf  |  R  | Master to slave FIFO full  flag
//      2  | S2MFe  |  R  | Slave to master FIFO empty flag
//      3  | S2MFf  |  R  | Slave to master FIFO full  flag
//      4  | Tunnel |  RW | 0/1 dis/enables M2S direct write
//      5  | Reset  |  W  | 1 resets the module
//
//   Direct write access from the master to the slave bus is possible if the
//   tunnel bit is set. This is mainly inteded for booting after reset and
//   should not be used during normal operation. The tunnel bit is active
//   after reset.
//
// HISTORY:
//   29-Jan-11, Dirk Sommer
//     Initial version
//
//====================================================================================

module bridge(irqMst,
              irqSlv,
              regWeSlvOut, 
              regBusMst, regWeMst, regReMst,
              regBusSlv, regWeSlv, regReSlv,
              clk, reset);

  // ---------------------------------------------------------------------------------
  // Module parameters - set addresses via defparam from outside
  parameter R_fifoMst         = 0;
  parameter R_ctrlMst         = 0;
  parameter R_fifoSlv         = 0;
  parameter R_ctrlSlv         = 0;

  // Control / status register bits
  parameter BitM2SFe_C        = 0;
  parameter BitM2SFf_C        = 1;
  parameter BitS2MFe_C        = 2;
  parameter BitS2MFf_C        = 3;
  parameter BitTunnel_C       = 4;
  parameter BitReset_C        = 5;
  
  // ---------------------------------------------------------------------------------
  // Inputs and outputs
  output                       irqMst;
  output                       irqSlv;

  // Inputs and outputs for registers
  inout                [15:0]  regBusMst;
  input                        regWeMst;
  input                        regReMst;
  inout                [15:0]  regBusSlv;
  input                        regWeSlv;
  input                        regReSlv;
  output                       regWeSlvOut;

  // Define clock and reset
  input                        clk;
  input                        reset;

  // ---------------------------------------------------------------------------------
  // Define wires and registers
  reg    [5:0] ctrl;
  wire         rstInt;
      
  reg          ctrlMstWe;
  reg          ctrlMstRe;
  reg          fifoMstWe;
  reg          fifoMstRe;
  reg          fifoMstWeD;
      
  reg          ctrlSlvWe;
  reg          ctrlSlvRe;
  reg          fifoSlvWe;
  reg          fifoSlvRe;
  reg          fifoSlvWeD;
  
  reg          regWeMstD;
  reg          regWeMstDD;
  reg   [15:0] regBusMstD;

  // ---------------------------------------------------------------------------------
  // Define internal reset
  assign rstInt = reset | ctrl[BitReset_C];
  
  // ---------------------------------------------------------------------------------
  // Define FIFOs

  // ----------------------------------------
  // Fifo master -> slave
  wire               [15:0]  ffM2S_q;
  wire                       ffM2S_fe;
  wire                       ffM2S_ff;
  wire                       ffM2S_ff2;
  wire                       ffM2S_fuf;
  wire                       ffM2S_fof;
  wire               [15:0]  ffM2S_d;
  wire                       ffM2S_re;
  wire                       ffM2S_we;
  wire                       ffM2S_ir;
  wire                       ffM2S_or;
  wire                       ffM2S_rst;

  fifoB ffM2S(ffM2S_q, ffM2S_fe, ffM2S_ff, ffM2S_ff2, ffM2S_fuf,
              ffM2S_fof, ffM2S_d, ffM2S_re, ffM2S_we, ffM2S_ir,
              ffM2S_or, ffM2S_rst, clk);
  defparam ffM2S.w = 16;
  defparam ffM2S.m = 5;
  defparam ffM2S.n = 16;
  defparam ffM2S.a = 0;

  assign ffM2S_d                   = regBusMst;
  assign ffM2S_we                  = fifoMstWe;
//  assign                           = ffM2S_ff;
  assign ffM2S_or                  = 0;

//  assign                           = ffM2S_q;
//  assign                           = ffM2S_fe;
  assign ffM2S_re                  = fifoSlvRe;
  assign ffM2S_ir                  = 0;

  assign ffM2S_rst                 = rstInt;

  // ----------------------------------------
  // Fifo slave -> master
  wire               [15:0]  ffS2M_q;
  wire                       ffS2M_fe;
  wire                       ffS2M_ff;
  wire                       ffS2M_ff2;
  wire                       ffS2M_fuf;
  wire                       ffS2M_fof;
  wire               [15:0]  ffS2M_d;
  wire                       ffS2M_re;
  wire                       ffS2M_we;
  wire                       ffS2M_ir;
  wire                       ffS2M_or;
  wire                       ffS2M_rst;

  fifoB ffS2M(ffS2M_q, ffS2M_fe, ffS2M_ff, ffS2M_ff2, ffS2M_fuf,
              ffS2M_fof, ffS2M_d, ffS2M_re, ffS2M_we, ffS2M_ir,
              ffS2M_or, ffS2M_rst, clk);
  defparam ffS2M.w = 16;
  defparam ffS2M.m = 5;
  defparam ffS2M.n = 16;
  defparam ffS2M.a = 0;

  assign ffS2M_d                   = regBusSlv;
  assign ffS2M_we                  = fifoSlvWe;
//  assign                           = ffS2M_ff;
  assign ffS2M_or                  = 0;

//  assign                           = ffS2M_q;
//  assign                           = ffS2M_fe;
  assign ffS2M_re                  = fifoMstRe;
  assign ffS2M_ir                  = 0;

  assign ffS2M_rst                 = rstInt;

  // ---------------------------------------------------------------------------------
  // Main FSM
  always @(posedge clk) begin
    
    // Reset
    if (rstInt) begin
      ctrl[BitM2SFf_C]  <= 0;
      ctrl[BitM2SFe_C]  <= 1;
      ctrl[BitS2MFf_C]  <= 0;
      ctrl[BitS2MFe_C]  <= 1;
      ctrl[BitTunnel_C] <= 1;
      ctrl[BitReset_C]  <= 0;

      ctrlMstWe         <= 0;
      ctrlMstRe         <= 0;
      fifoMstWe         <= 0;
      fifoMstRe         <= 0;
      fifoMstWeD        <= 0;
      
      ctrlSlvWe         <= 0;
      ctrlSlvRe         <= 0;
      fifoSlvWe         <= 0;
      fifoSlvRe         <= 0;
      fifoSlvWeD        <= 0;

      regWeMstD         <= 0;
      regWeMstDD        <= 0;
      regBusMstD        <= 0;
    
    end else begin
      
      // Delayed write and read enable via regBus (mst or slv)
      ctrlMstWe  <= regBusMst==R_ctrlMst & regWeMst;
      ctrlMstRe  <= regBusMst==R_ctrlMst & regReMst;
      fifoMstWe  <= regBusMst==R_fifoMst & regWeMst;
      fifoMstRe  <= regBusMst==R_fifoMst & regReMst;
      fifoMstWeD <= fifoMstWe;
      
      ctrlSlvWe  <= regBusSlv==R_ctrlSlv & regWeSlv;
      ctrlSlvRe  <= regBusSlv==R_ctrlSlv & regReSlv;
      fifoSlvWe  <= regBusSlv==R_fifoSlv & regWeSlv;
      fifoSlvRe  <= regBusSlv==R_fifoSlv & regReSlv;
      fifoSlvWeD <= fifoSlvWe;
      
      // Update FF/FE flags in ctrl register
      ctrl[BitM2SFf_C] <= ffM2S_ff;
      ctrl[BitM2SFe_C] <= ffM2S_fe;
      ctrl[BitS2MFf_C] <= ffS2M_ff;
      ctrl[BitS2MFe_C] <= ffS2M_fe;
      
      // Possibly en/disable tunnel or reset     
      if (ctrlMstWe) begin
        ctrl[BitTunnel_C] <= regBusMst[BitTunnel_C];
        ctrl[BitReset_C]  <= regBusMst[BitReset_C];
      end else
      
      if (ctrlSlvWe) begin
        ctrl[BitTunnel_C] <= regBusSlv[BitTunnel_C];
        ctrl[BitReset_C]  <= regBusSlv[BitReset_C];
      end                
    
      // Registers for tunnel write from master to slave
      regWeMstD   <= regWeMst;
      regWeMstDD  <= regWeMstD;
      regBusMstD  <= regBusMst;
    
    end // if (rstInt) ...      
  end //always

  // ---------------------------------------------------------------------------------
  // Assign regBus output
  assign regBusMst  = (ctrlMstRe & ~rstInt) ? {10'b0000000000, ctrl} :  
                      (fifoMstRe & ~rstInt) ? ffS2M_q :  
                      {16{1'bz}};
  assign regBusSlv  = (ctrlSlvRe & ~rstInt) ? {10'b0000000000, ctrl} :  
                      (fifoSlvRe & ~rstInt) ? ffM2S_q :  
                      ((regWeMstD | regWeMstDD) & ctrl[BitTunnel_C] & ~rstInt) ? regBusMstD :
                      {16{1'bz}};
  assign regWeSlvOut = regWeMstD & ctrl[BitTunnel_C] & ~rstInt;

  // ---------------------------------------------------------------------------------
  // Assign IRQ outputs
  assign  irqMst = fifoSlvWe | fifoSlvWeD | ~ctrl[BitS2MFe_C];
  assign  irqSlv = fifoMstWe | fifoMstWeD | ~ctrl[BitM2SFe_C];

endmodule
// ===================================================================================
