//====================================================================================
//  Copyright (C) BAY9, 2015
//====================================================================================
//
// MODULE:
//   rx11aTest
//
// PURPOSE:
//   This is test module for rx11a, including the FFT
//
// INPUT:
//   x:       RX input data at 20 MHz
//
// OUTPUT:
//   y:       Decoded output bytes
//   yAcqCor: ACQ SS CCF output
//   yAcqAbs: ACQ CTFE abs output
//   yAcqPh:  ACQ CTFE phase output
//   ySyncAbs:FTE abs outputs
//   ySyncC:  Sync channel corrected outputs
//   ySyncSh: Sync shift factor outputs
//   yDemD:   Demod data output
//   yDemSb:  Demod soft bit output
//   yDecSb:  Decoder soft bit output after deinterleaver
//   yDecSb2: Decoder soft bit output after depuncturer
//   yDecVit: Viterbi output bytes
//
// PARAMETER:
//   wx:      Input bit width
//   wf:      FFT input/output bit width
//   modeSyncFte:Fine timing estimation
//   modeSyncFfe:Fine frequency estimation
//   modeSyncAmpEst:Amplitude estimation
//   modeSyncChEst:Channel estimation
//   modeSyncChCor:Channel correction
//
// DESCRIPTION:
//   Schematic:
//           .....................         
//           :                   :         
//           :      ||    :         
//           :   .->| fft64 |-.  :         
//           :   |  |_______| |  :         
//           :   |            |  :         
//           :   `----.   .---  :         
//           :    yFft|   |xFft  :         
//           :        |   |      :         
//        x  :      ||    : b       
//       -----------| rx11a |---------->   
//           :      |_______|    :         
//           :           |       :         
//           : rx11aTest |       :         
//           :...........|.......:         
//                       |                 
//                       `debug outputs    
//   
//
// HISTORY:
//   21-Sep-10, Dirk Sommer
//     Initial version
//
//====================================================================================

module rx11aTest(y, y_or, y_ff, y_we, 
                 yAcqCor, yAcqCor_or, yAcqCor_ff, yAcqCor_we, 
                 yAcqAbs, yAcqAbs_or, yAcqAbs_ff, yAcqAbs_we, 
                 yAcqPh, yAcqPh_or, yAcqPh_ff, yAcqPh_we, 
                 ySyncAbs, ySyncAbs_or, ySyncAbs_ff, ySyncAbs_we, 
                 ySyncC, ySyncC_or, ySyncC_ff, ySyncC_we, 
                 ySyncSh, ySyncSh_or, ySyncSh_ff, ySyncSh_we, 
                 yDemD, yDemD_or, yDemD_ff, yDemD_we, 
                 yDemSb, yDemSb_or, yDemSb_ff, yDemSb_we, 
                 yDecSb, yDecSb_or, yDecSb_ff, yDecSb_we, 
                 yDecSb2, yDecSb2_or, yDecSb2_ff, yDecSb2_we, 
                 yDecVit, yDecVit_or, yDecVit_ff, yDecVit_we, 
                 x, x_ir, x_fe, x_re, 
                 regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter y_w               = 0;
  parameter yAcqCor_w         = 0;
  parameter yAcqAbs_w         = 0;
  parameter yAcqPh_w          = 0;
  parameter ySyncAbs_w        = 0;
  parameter ySyncC_w          = 0;
  parameter ySyncSh_w         = 0;
  parameter yDemD_w           = 0;
  parameter yDemSb_w          = 0;
  parameter yDecSb_w          = 0;
  parameter yDecSb2_w         = 0;
  parameter yDecVit_w         = 0;
  parameter x_w               = 0;


  parameter wx                = 0;
  parameter wf                = 0;
  parameter modeSyncFte       = 0;
  parameter modeSyncFfe       = 0;
  parameter modeSyncAmpEst    = 0;
  parameter modeSyncChEst     = 0;
  parameter modeSyncChCor     = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output            [y_w-1:0]  y;
  output                       y_or;
  input                        y_ff;
  output                       y_we;
  assign                       y_we = y_or & ~y_ff;

  output      [yAcqCor_w-1:0]  yAcqCor;
  output                       yAcqCor_or;
  input                        yAcqCor_ff;
  output                       yAcqCor_we;
  assign                       yAcqCor_we = yAcqCor_or & ~yAcqCor_ff;

  output      [yAcqAbs_w-1:0]  yAcqAbs;
  output                       yAcqAbs_or;
  input                        yAcqAbs_ff;
  output                       yAcqAbs_we;
  assign                       yAcqAbs_we = yAcqAbs_or & ~yAcqAbs_ff;

  output       [yAcqPh_w-1:0]  yAcqPh;
  output                       yAcqPh_or;
  input                        yAcqPh_ff;
  output                       yAcqPh_we;
  assign                       yAcqPh_we = yAcqPh_or & ~yAcqPh_ff;

  output     [ySyncAbs_w-1:0]  ySyncAbs;
  output                       ySyncAbs_or;
  input                        ySyncAbs_ff;
  output                       ySyncAbs_we;
  assign                       ySyncAbs_we = ySyncAbs_or & ~ySyncAbs_ff;

  output       [ySyncC_w-1:0]  ySyncC;
  output                       ySyncC_or;
  input                        ySyncC_ff;
  output                       ySyncC_we;
  assign                       ySyncC_we = ySyncC_or & ~ySyncC_ff;

  output      [ySyncSh_w-1:0]  ySyncSh;
  output                       ySyncSh_or;
  input                        ySyncSh_ff;
  output                       ySyncSh_we;
  assign                       ySyncSh_we = ySyncSh_or & ~ySyncSh_ff;

  output        [yDemD_w-1:0]  yDemD;
  output                       yDemD_or;
  input                        yDemD_ff;
  output                       yDemD_we;
  assign                       yDemD_we = yDemD_or & ~yDemD_ff;

  output       [yDemSb_w-1:0]  yDemSb;
  output                       yDemSb_or;
  input                        yDemSb_ff;
  output                       yDemSb_we;
  assign                       yDemSb_we = yDemSb_or & ~yDemSb_ff;

  output       [yDecSb_w-1:0]  yDecSb;
  output                       yDecSb_or;
  input                        yDecSb_ff;
  output                       yDecSb_we;
  assign                       yDecSb_we = yDecSb_or & ~yDecSb_ff;

  output      [yDecSb2_w-1:0]  yDecSb2;
  output                       yDecSb2_or;
  input                        yDecSb2_ff;
  output                       yDecSb2_we;
  assign                       yDecSb2_we = yDecSb2_or & ~yDecSb2_ff;

  output      [yDecVit_w-1:0]  yDecVit;
  output                       yDecVit_or;
  input                        yDecVit_ff;
  output                       yDecVit_we;
  assign                       yDecVit_we = yDecVit_or & ~yDecVit_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;


  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  // --------------------------------------------------------------------------------------
  // Included instances
  `include "rx11a_0.v"
  `include "fft64_0.v"

  // --------------------------------------------------------------------------------------
  // Define connections
  assign rx11a_0_x                 = x;
  assign rx11a_0_x_fe              = x_fe;
  assign x_ir                      = rx11a_0_x_ir;

  assign fft64_0_x                 = rx11a_0_yFft;
  assign fft64_0_x_fe              = ~rx11a_0_yFft_or;
  assign rx11a_0_yFft_ff           = ~fft64_0_x_ir;

  assign rx11a_0_xFft              = fft64_0_y;
  assign rx11a_0_xFft_fe           = ~fft64_0_y_or;
  assign fft64_0_y_ff              = ~rx11a_0_xFft_ir;

  assign y                         = rx11a_0_y;
  assign y_or                      = rx11a_0_y_or;
  assign rx11a_0_y_ff              = y_ff;

  assign yAcqCor                   = rx11a_0_yAcqCor;
  assign yAcqCor_or                = rx11a_0_yAcqCor_or;
  assign rx11a_0_yAcqCor_ff        = yAcqCor_ff;

  assign yAcqPh                    = rx11a_0_yAcqPh;
  assign yAcqPh_or                 = rx11a_0_yAcqPh_or;
  assign rx11a_0_yAcqPh_ff         = yAcqPh_ff;

  assign yAcqAbs                   = rx11a_0_yAcqAbs;
  assign yAcqAbs_or                = rx11a_0_yAcqAbs_or;
  assign rx11a_0_yAcqAbs_ff        = yAcqAbs_ff;

  assign ySyncAbs                  = rx11a_0_ySyncAbs;
  assign ySyncAbs_or               = rx11a_0_ySyncAbs_or;
  assign rx11a_0_ySyncAbs_ff       = ySyncAbs_ff;

  assign ySyncC                    = rx11a_0_ySyncC;
  assign ySyncC_or                 = rx11a_0_ySyncC_or;
  assign rx11a_0_ySyncC_ff         = ySyncC_ff;

  assign ySyncSh                   = rx11a_0_ySyncSh;
  assign ySyncSh_or                = rx11a_0_ySyncSh_or;
  assign rx11a_0_ySyncSh_ff        = ySyncSh_ff;

  assign yDemD                     = rx11a_0_yDemD;
  assign yDemD_or                  = rx11a_0_yDemD_or;
  assign rx11a_0_yDemD_ff          = yDemD_ff;

  assign yDemSb                    = rx11a_0_yDemSb;
  assign yDemSb_or                 = rx11a_0_yDemSb_or;
  assign rx11a_0_yDemSb_ff         = yDemSb_ff;

  assign yDecSb                    = rx11a_0_yDecSb;
  assign yDecSb_or                 = rx11a_0_yDecSb_or;
  assign rx11a_0_yDecSb_ff         = yDecSb_ff;

  assign yDecSb2                   = rx11a_0_yDecSb2;
  assign yDecSb2_or                = rx11a_0_yDecSb2_or;
  assign rx11a_0_yDecSb2_ff        = yDecSb2_ff;

  assign yDecVit                   = rx11a_0_yDecVit;
  assign yDecVit_or                = rx11a_0_yDecVit_or;
  assign rx11a_0_yDecVit_ff        = yDecVit_ff;


  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | rx11a_0_regWe | fft64_0_regWe;
  assign regReOut = 0 | rx11a_0_regRe | fft64_0_regRe;

// ----------------------------------------------------------------------------------------

//=========================================================================================
// Autogenerate end - do not remove this comment
//=========================================================================================

  // Don't use ACQ halt during testing
  assign rx11a_0_ssAcqHalt = 0;

endmodule
//=========================================================================================

