function stat = testRx11g(id, numFrames, rxTimeSec, printResults)
%===============================================================================
%  Copyright (C) BAY9, 2016
%===============================================================================
%
% Syntax:
%   testRx11g(id, numFrames, rxTimeSec, printResults)
%
% Inputs:
%   id            - Target identifier
%   numFrames     - Number of frames to receive
%   rxTimeSec     - Time to wait for an RX frame
%   printResults  - 0/1 -> do not / do print results statistics
%
% Outputs:
%   stat          - Error statistic, struct with fields
%                    .crcOk
%                    .crcErr
%                    .hdrErr
%                    .noAcq
%
% Description:
%   Try rxTimeSec maximum to receive numFrames frames. Basic reception
%   parameters are printed. The function sends an RxReq message to the target
%   and waits rxTimeSec for each frame. If a frame is received, some basic
%   data of the receved frame is printed. After reading the RX confirm
%   messages, the followingdata is displayed
%       o M:      RX mode (0: BPSK R=1/2, 1: BPSK R=3/4, ..., 7: 64-QAM R=3/4)
%       o Len:    Length in bytes
%       o Amp:    Amplitude input level
%       o AGC:    AGC step taken (roughly attenuation in dB)
%       o fOffHz: Measured frequency offset in Hz
%       o C:      CRC status 
%                   0 = CRC OK
%                   1 = CR Error
%                   2 = Header error
%                   3 = Acquisition failed, no RX frame
%

  % ------------------------------------------------------------------
  % Default idStd
  global idStd;
  if (isempty(idStd))
    idStd.tgt   = 'fpga0';
    idStd.top   = 'wlanX';
    idStd.inst  = 0;
  end

  % ----------------------------------------------------------------------------
  % Check input arguments, possible set some defaults
  if (nargin < 1 || isempty(id))
    id = idStd;
  end
  if (nargin < 2 || isempty(numFrames))
    numFrames = 10;
  end
  if (nargin < 3 || isempty(rxTimeSec))
    rxTimeSec = 1;
  end
  if (nargin < 4 || isempty(printResults))
    printResults = 1;
  end
      
  % ----------------------------------------------------------------------------
  % RX timing
  rxTime32    = round(rxTimeSec*20e6);
  rxTimerMode = 1;

  % Keep some statistics
  crcAll = zeros(1, numFrames);

  % ----------------------------------------------------------------------------
  % Main burst loop
  for k = 0:numFrames-1

    % --------------------------------------------    
    % Send initial RX start request + read start cfm
    cfmRxStart  = msgWlanRxRun1(id, rxTimerMode, rxTime32);
    
    % --------------------------------------------
    % Evaluate remaining RX confirm messages
    [cfmRxAcqEnd, cfmRxHdr, cfmRxEnd] = msgWlanRxRun2(id);

    % --------------------------------------------------------------------------
    % Evaluate RX return parameters
    [rxp, rxpS] = ...
      testRxReqEval(cfmRxAcqEnd, cfmRxHdr, cfmRxEnd);
    
    % Keep CRC errors
    crcAll(k+1) = rxp.crc;
    
    % --------------------------------------------------------------------------
    % Print header each 20 frames
    if (mod(k, 20)==0)
      fprintf('\n');
      fprintf('+--------+---+------+-----+-----+---------+---+\n');
      fprintf('|   No   | M |  Len | Amp | AGC |  fOffHz | C |\n');
      fprintf('+--------+---+------+-----+-----+---------+---+\n');
    end
    fprintf('| %6d | %s | %s | %s | %s | %s | %s |\n', ...
              k, rxpS.rxMode, rxpS.rxLen, rxpS.amp, rxpS.agcAttn, ...
                 rxpS.fOffHz, rxpS.crc);
  end
  
  % --------------------------------------------------------------------------
  % Calculate statistics
  stat.crcOk  = sum(crcAll==0) / numFrames;
  stat.crcErr = sum(crcAll==1) / numFrames;
  stat.hdrErr = sum(crcAll==2) / numFrames;
  stat.noAcq  = sum(crcAll==3) / numFrames;

  if (printResults)
    fprintf('------------------------------------------------------\n');
    fprintf('\n');
    fprintf('  CRC OK:  %5.1f%%\n', stat.crcOk *100);
    fprintf('  CRC Err: %5.1f%%\n', stat.crcErr*100);
    fprintf('  Hdr Err: %5.1f%%\n', stat.hdrErr*100);
    fprintf('  No  Acq: %5.1f%%\n', stat.noAcq *100);
    fprintf('\n');
  end

% ------------------------------------------------------------------------------
