//====================================================================================
//  Copyright (C) BAY9, 2014
//====================================================================================
//
// MODULE:
//   wlan
//
// PURPOSE:
//   Top module
//
// INPUT/OUTPUT:
//   Clock/reset:
//     clk:           in:  Clock
//     resetIn:       in:  Reset (active high)
//     resetOut:      out: Indicates that system is still in reset
//   ADC/DAC:
//     adcRe/Im:      in:  ADC inputs, real/imag part, 12 bit signed
//     adc_or:        in:  ADC handshake (input), adc_or=1 indicates valid data at adcRe/Im
//     dacRe/Im:      out: DAC outputs, real/imag part, 12 bit signed
//     dac_ir:        in:  DAC handshake (input), dac_ir=1 indicates data to be read from dacRe/Im
//   RF control:
//     attn:          out: Attenuation lines (16 parallel outputs) to control RF gain
//     attn_or:       out: Output ready signal, attn_or=1 -> new data at attn
//     paOn:          out: Power amplifier on (paOn=1)
//     txOn:          out: Transmitter on (txOn=1)
//     rxOn:          out: Receiver on (rxOn=1)
//     twbClk:        out: Three wire bus clock
//     twbDat:        out: Three wire bus data
//     twbEn:         out: Three wire bus enable
//   Data input/output:
//     dataTx:        in:  Transmitter data input
//     dataTx_ir:     out: Input ready, indicates data being read from host
//     dataRx:        out: Receiver data output
//     dataRx_or:     out: Output ready, indicates data being written to host
//   Control input/output: 
//     ctrlSel:       in:  Control select, 0=CtrlIf, 1=UART
//     uartOut:       out: UART transmit signal
//     uartIn:        in:  UART receive signal
//     ctrlOut:       out: External control interface (16 bit), transmit signal
//     ctrlOut_or:    out: Output ready signal
//     ctrlOut_ir:    in:  Input ready signal
//     ctrlIn:        in:  External control interface (16 bit), receive signal
//     ctrlIn_or:     in:  Output ready signal
//     ctrlIn_ir:     out: Input ready signal
//   Other:    
//     gpo:           out: General purpose output (16 bit)
//     gpi:           in:  General purpose input (16 bit)
//     statLed:       out: Status LEDs (8 bit)
//
// DESCRIPTION:
//   Module wlan.v provide a wrapper for the autogenerated module main.v with
//   the following functionality:
//    - Order inputs/output according to functionality
//    - Register RX-ADC input
//    - Include auto-reset circuit (resetCnt)
//    - Provide register bus control feedback
//
// HISTORY:
//   17-Nov-13, Dirk Sommer
//     Initial version
//
//====================================================================================

module wlan(clk,
            resetIn,
            resetOut,
            
            adcRe,
            adcIm,
            adc_or,

            dacRe,
            dacIm,
            dac_ir,
            
            attn,
            attn_or,
            
            paOn,
            txOn,
            rxOn,
            
            twbClk, twbDat, twbEn,
            
            dataTx, dataTx_ir, dataTx_or,
            dataRx, dataRx_or,
            
            ctrlSel,
            uartOut,
            uartIn,
            ctrlOut, ctrlOut_or, ctrlOut_ir,
            ctrlIn,  ctrlIn_or,  ctrlIn_ir,

            gpo, 
            gpi, 
            
            statLed);

  // --------------------------------------------------------------------------------------
  // Include global definitions
  `include "def_Const_wlan.v"

  // Clock, reset
  input                             clk;
  input                             resetIn;  
  output                            resetOut; 
                                 
  // ADC/DAC
  input    [WlanAdcDacWidth_C-1:0]  adcRe;
  input    [WlanAdcDacWidth_C-1:0]  adcIm;
  input                             adc_or;
  output   [WlanAdcDacWidth_C-1:0]  dacRe;
  output   [WlanAdcDacWidth_C-1:0]  dacIm;
  input                             dac_ir;

  // RF control
  output  [WlanAttnSelWidth_C-1:0]  attn;
  output                            attn_or;

  output                            paOn;
  output                            txOn;
  output                            rxOn;
  
  output                            twbClk;
  output                            twbDat;
  output                            twbEn;
  
  // Data in/out
  input      [WlanDataWidth_C-1:0]  dataTx; 
  output                            dataTx_ir;
  input                             dataTx_or;
  output     [WlanDataWidth_C-1:0]  dataRx;
  output                            dataRx_or;

  // Control select, UART, external interface
  input                             ctrlSel;
  
  output                            uartOut;
  input                             uartIn;
  
  output      [WlanCtrlWidth_C-1:0] ctrlOut;
  output                            ctrlOut_or;
  input                             ctrlOut_ir;
  input       [WlanCtrlWidth_C-1:0] ctrlIn;
  input                             ctrlIn_or;
  output                            ctrlIn_ir;

  // Status LEDs + external bus + GPIO
  output     [WlanGpioWidth_C-1:0]  gpo;
  input      [WlanGpioWidth_C-1:0]  gpi;
  output  [WlanStatLedWidth_C-1:0]  statLed;

  // ----------------------------------------
  // Unused clock signals
  wire                          clk2;

  // RegBus signals
  wire                   [15:0] regBus;
  wire                          regWe;
  wire                          regRe;
 
  // Reset lines
  wire                          reset;

  // Register ADC input
  reg   [WlanAdcDacWidth_C-1:0] adcReR;
  reg   [WlanAdcDacWidth_C-1:0] adcImR;
  reg                           adcR_or;

  // ===========================================================================
  // Define main module
  `include "main_0.v"

  // ---------------------------------------------------------------------------
  // Set unused clock signals to 0
  assign clk2                 = 0;
  
  // ---------------------------------------------------------------------------
  // Register bus controls feedback
  assign regWe                = main_0_regWe;
  assign regRe                = main_0_regRe;

  // ---------------------------------------------------------------------------
  // Define ADC/DAC connections
  always @(posedge clk) begin       // Register ADC input, just in case...
    if (reset) begin
      adcReR  <= 0;
      adcImR  <= 0;
      adcR_or <= 0;
    end else begin    
      adcReR  <= adcRe;
      adcImR  <= adcIm;
      adcR_or <= adc_or;
    end
  end

  assign main_0_adcRe         = adcReR;
  assign main_0_adcIm         = adcImR;
  assign main_0_adcRe_fe      = ~adcR_or;
  assign main_0_adcIm_fe      = ~adcR_or;

  assign dacRe                = main_0_dacRe; 
  assign dacIm                = main_0_dacIm; 
  assign main_0_dacRe_ff      = ~dac_ir;     
  assign main_0_dacIm_ff      = ~dac_ir;     

  // ---------------------------------------------------------------------------
  // Define RF connections
  assign attn                 = main_0_attn;
  assign attn_or              = main_0_attn_or;

  assign paOn                 = main_0_paOn;
  assign txOn                 = main_0_txOn;
  assign rxOn                 = main_0_rxOn;
  
  assign twbClk               = main_0_twbClk;
  assign twbDat               = main_0_twbDat;
  assign twbEn                = main_0_twbEn;
  
  // ---------------------------------------------------------------------------
  // Define byte input / output connections
  assign main_0_dataTx        = dataTx;
  assign dataTx_ir            = main_0_dataTx_ir;
  assign main_0_dataTx_fe     = ~dataTx_or;
  
  assign dataRx               = main_0_dataRx;
  assign dataRx_or            = main_0_dataRx_or;
  assign main_0_dataRx_ff     = 0;

  // ---------------------------------------------------------------------------
  // Connect UART, control interface, and control selector
  assign  main_0_ctrlSel      = ctrlSel;
  
  assign  uartOut             = main_0_uartOut;
  assign  main_0_uartIn       = uartIn;

  assign  ctrlOut             = main_0_ctrlOut;    
  assign  ctrlOut_or          = main_0_ctrlOut_or; 
  assign  main_0_ctrlOut_ff   = ~ctrlOut_ir;       
  
  assign  main_0_ctrlIn       = ctrlIn;            
  assign  ctrlIn_ir           = main_0_ctrlIn_ir;  
  assign  main_0_ctrlIn_fe    = ~ctrlIn_or;        
  
  // ---------------------------------------------------------------------------
  // Connect status LEDs, GPO, and GPI
  assign gpo                  = main_0_gpo;
  assign main_0_gpi           = gpi;
  assign statLed              = main_0_statLed;

  // ----------------------------------------
  // Reset circuit for power on and reset combine
  resetCnt resetCnt(reset, main_0_resetOut | resetIn, clk);
  assign resetOut = reset;

endmodule

//==============================================================================

