//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   src11to20
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module src11to20(x20, x20_or, x20_ff, x20_we, 
                 x11, x11_ir, x11_fe, x11_re, 
                 run1, 
                 regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter x20_w             = 0;
  parameter x11_w             = 0;

  parameter run1_w            = 0;

  parameter w                 = 0;
  parameter ldPatInit         = 0;

  parameter x11Re_w           = 0;
  parameter x11Re_n           = 0;
  parameter x11Re_m           = 0;
  parameter x11Im_w           = 0;
  parameter x11Im_n           = 0;
  parameter x11Im_m           = 0;
  parameter sumX11Re_w        = 0;
  parameter sumX11Re_n        = 0;
  parameter sumX11Re_m        = 0;
  parameter sumX11Im_w        = 0;
  parameter sumX11Im_n        = 0;
  parameter sumX11Im_m        = 0;
  parameter cnt_w             = 0;
  parameter cnt_n             = 0;
  parameter cnt_m             = 0;
  parameter ldPat_w           = 0;
  parameter ldPat_n           = 0;
  parameter ldPat_m           = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output          [x20_w-1:0]  x20;
  output                       x20_or;
  input                        x20_ff;
  output                       x20_we;
  assign                       x20_we = x20_or & ~x20_ff;

  input           [x11_w-1:0]  x11;
  output                       x11_ir;
  input                        x11_fe;
  output                       x11_re;
  assign                       x11_re = x11_ir & ~x11_fe;

  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  // --------------------------------------------------------------------------------------
  // Internal status registers
  reg           [x11Re_w-1:0]  x11Re[x11Re_n-1:0];
  reg           [x11Im_w-1:0]  x11Im[x11Im_n-1:0];
  reg        [sumX11Re_w-1:0]  sumX11Re;
  reg        [sumX11Im_w-1:0]  sumX11Im;
  reg             [cnt_w-1:0]  cnt;
  reg           [ldPat_w-1:0]  ldPat;

  // --------------------------------------------------------------------------------------
  // Included instances
  `include "lut_cf11bTx.v"

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | lut_cf11bTx_regWe;
  assign regReOut = 0 | lut_cf11bTx_regRe;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // Internal registers and variables
  wire            [w-1:0] cf[3:0];      // Current coefficients
  wire              [1:0] x11S[3:0];    //
  wire            [w-1:0] x11M[3:0];    //
  wire            [w-1:0] sumX11M;
  wire              [1:0] x11ReIn, x11ImIn;

  // FSM status registers
  wire       [cnt_w-1:0]  cntNext;      // cnt from 0..79
  wire     [ldPat_w-1:0]  ldPatNext;    // load pattern for SRC
  wire                    runFsm;       // FSM is running

  // Stop the machine if no input data available, but data is requested at the output
  assign run1_d     = 0;
  assign run1_weint = x11_fe & ~x20_ff;
  
  // Debug variables
  wire   [w-1:0] cf0, cf1, cf2, cf3;
  assign cf0 = cf[0];
  assign cf1 = cf[1];
  assign cf2 = cf[2];
  assign cf3 = cf[3];
  
  wire   [1:0] x11S0, x11S1, x11S2, x11S3;
  assign x11S0 = x11S[0];
  assign x11S1 = x11S[1];
  assign x11S2 = x11S[2];
  assign x11S3 = x11S[3];

  wire   [w-1:0] x11M0, x11M1, x11M2, x11M3;
  assign x11M0 = x11M[0];
  assign x11M1 = x11M[1];
  assign x11M2 = x11M[2];
  assign x11M3 = x11M[3];

  // -------------------------------------------------------------------------------------
  // Input buffer read enable
  assign x11_ir = cnt[0]==1 & ldPat[0]==1;
  assign x20_or = cnt[0]==0 & run1==1;
  assign x20    = {sumX11Im, sumX11Re};

  // FSM-feedback
  assign cntNext    = (cnt==39) ? 0 : cnt+1;        // count from 0..39
  assign ldPatNext  = {ldPat[0], ldPat[19:1]};      // rotate shift pattern

  // Signal processing
  assign lut_cf11bTx_adr = cntNext[cnt_w-1:1];      // Assign LUT address

  assign x11ReIn = x11[1:0];  // Define input, use new input
  assign x11ImIn = x11[3:2];  // immediately or saved input
  
  assign cf[0] = lut_cf11bTx_dat[1*w-1:0*w];        // Split LUT into coefficients
  assign cf[1] = lut_cf11bTx_dat[2*w-1:1*w];
  assign cf[2] = lut_cf11bTx_dat[3*w-1:2*w];
  assign cf[3] = lut_cf11bTx_dat[4*w-1:3*w];

  assign x11S[0] = cnt[0]==0 ? x11Re[0] : x11Im[0]; // Select real/imag part as input
  assign x11S[1] = cnt[0]==0 ? x11Re[1] : x11Im[1]; // to the -1/0/+1 multiplication
  assign x11S[2] = cnt[0]==0 ? x11Re[2] : x11Im[2];
  assign x11S[3] = cnt[0]==0 ? x11ReIn  : x11ImIn;
  
  assign x11M[0] = x11S[0]==0 ? 0 : (x11S[0]==1 ? cf[0] : -cf[0]); // Select -1/0/+1
  assign x11M[1] = x11S[1]==0 ? 0 : (x11S[1]==1 ? cf[1] : -cf[1]); // as input to the
  assign x11M[2] = x11S[2]==0 ? 0 : (x11S[2]==1 ? cf[2] : -cf[2]); // final adder
  assign x11M[3] = x11S[3]==0 ? 0 : (x11S[3]==1 ? cf[3] : -cf[3]);  
  
  assign sumX11M = x11M[0] + x11M[1] + x11M[2] + x11M[3]; // Add up multiplied cf values
  
  //----------------------------------------------------------------------------------------
  // Register update
  always @(posedge clk) begin
    
    // Reset
    if (run1==0) begin
      cnt       <= 0;
      ldPat     <= ldPatInit;
      x11Re[0]  <= 0;
      x11Im[0]  <= 0;
      x11Re[1]  <= 0;
      x11Im[1]  <= 0;
      x11Re[2]  <= 0;
      x11Im[2]  <= 0;
      sumX11Re  <= 0;
      sumX11Im  <= 0;
    end else 

    // Main FSM
    if (~x11_fe & x20_we | cnt[0]==1) begin // Wait at cnt[0]==0 until data is
      cnt  <= cntNext;                      // read at the output, stop also if
      if (cnt[0]==0) begin                  // no input data available
        sumX11Re    <= sumX11M;       // Calculate next real output value
      end else begin
        sumX11Im    <= sumX11M;       // Calculate next imag output value
        ldPat       <= ldPatNext;     // Advance load pattern
        if (ldPat[0]==1) begin
          x11Re[0]  <= x11Re[1];      // Load next input sample if the load
          x11Im[0]  <= x11Im[1];      // pattern requires it and shift the
          x11Re[1]  <= x11Re[2];      // input data
          x11Im[1]  <= x11Im[2];
          x11Re[2]  <= x11ReIn;
          x11Im[2]  <= x11ImIn;
        end
      end
    end // if (runFsm)
  end // always @(posedge clk)

endmodule
//=========================================================================================
