//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   sbCalc
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module sbCalc(sb, sb_or, sb_ff, sb_we, 
              x, x_ir, x_fe, x_re, 
              sh, sh_ir, sh_fe, sh_re, 
              run1, 
              regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter sb_w              = 0;
  parameter x_w               = 0;
  parameter sh_w              = 0;

  parameter run1_w            = 0;

  parameter wx                = 0;
  parameter ws                = 0;
  parameter wb                = 0;
  parameter wbb               = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output           [sb_w-1:0]  sb;
  output                       sb_or;
  input                        sb_ff;
  output                       sb_we;
  assign                       sb_we = sb_or & ~sb_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  input            [sh_w-1:0]  sh;
  output                       sh_ir;
  input                        sh_fe;
  output                       sh_re;
  assign                       sh_re = sh_ir & ~sh_fe;

  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  // --------------------------------------------------------------------------------------
  // Included instances
  `include "buf1_22.v"

  // --------------------------------------------------------------------------------------
  // Define connections
  assign sb                        = buf1_22_y;
  assign sb_or                     = buf1_22_y_or;
  assign buf1_22_y_ff              = sb_ff;


  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | buf1_22_regWe;
  assign regReOut = 0 | buf1_22_regRe;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Reg+wire definitions for signal processing
  reg     [wx-1:0]  xS2;            // Sign input x>>2
  reg     [wx-1:0]  sb0Bp;          // x - x>>2
  reg     [wx-1:0]  sb0Br;          // x - x>>2 + 8
  reg     [wx-5:0]  sb0Bs;          // (x - x>>2 + 8) >> 4
    
  reg     [ws-1:0]  shFac;          // Currently used shift factor
  reg     [wx-1:0]  sb0p;           // SB0 before rndAdd/shift/sat (=x)
  reg     [wx-2:0]  sb0pAbs;        // Abs(sb0p)
  reg     [wx-1:0]  sb1p;           // SB1 before rndAdd/shift/sat
  reg     [wx-2:0]  sb1pAbs;        // Abs(sb1p)
  reg     [wx-1:0]  sb2p;           // SB2 before rndAdd/shift/sat
    
  reg     [wx-1:0]  sb0r;           // SB0 after rndAdd, before shift/sat
  reg     [wx-1:0]  sb1r;           // SB1 after rndAdd, before shift/sat
  reg     [wx-1:0]  sb2r;           // SB2 after rndAdd, before shift/sat
  
  reg     [wx-3:0]  sb0s;           // SB0 after rndAdd/shift, before sat
  reg     [wx-3:0]  sb1s;           // SB1 after rndAdd/shift, before sat
  reg     [wx-3:0]  sb2s;           // SB2 after rndAdd/shift, before sat
  
  wire    [wb-1:0]  sb0, sb1, sb2;  // Final SB0-2 output for QPSK/16-QAM/64-QAM
  wire   [wbb-1:0]  sb0B;           // Final Sb0   output for BPSK
  
  // Control (real registers)
  reg               cntIn;          // Counter for shift input data
  reg     [ws-1:0]  shD;            // Saved shift input
  
  // --------------------------------------------------------------------------------------
  // Assign input and output enable
//  assign x_ir  = run1 & ~sb_ff;
  assign x_ir  = run1 & buf1_22_x_ir;
  assign sh_ir = run1 & x_re & cntIn==0;
//  assign sb_or = run1 & ~x_fe;
  assign buf1_22_x_fe = ~(run1 & ~x_fe);
  
  // Reset line of buf1_22
  assign buf1_22_run1 = run1;

  // --------------------------------------------------------------------------------------
  // Save input shift factor value, to be used twice
  always @(posedge clk) begin

    // Reset input counter
    if (run1==0) begin
      cntIn <= 0;
      shD   <= 0;
    end else
    
    // Count input data (0,1)
    // Save shift input whenever cnt==0 and x_re=1
    begin
      if (x_re==1) begin
        cntIn <= cntIn+1;
        if (cntIn==0) begin
          shD <= sh;
        end
      end
    end

  end
  
  // --------------------------------------------------------------------------------------
  // Signal processing behaviour
  always @(*) begin
    
    // Calculate soft bits for BPSK    
    xS2     = {{2{x[wx-1]}}, x[wx-1:2]};
    sb0Bp   = x - xS2;                    // -x>>2
    sb0Br   = sb0Bp + 8;                  // +8
    sb0Bs   = sb0Br[wx-1:4];              // >>4
        
    // Calculate soft bits for QPSK, 16-QAM, 64-QAM
    shFac   = cntIn==0 ? sh : shD;                        // Input or saved shift factor
    sb0p    = x;                                          // sb0p = input x
    sb0pAbs = sb0p[wx-1] ? -sb0p[wx-2:0] : sb0p[wx-2:0];
    sb1p    = {2'b00, shFac} - sb0pAbs;                   // sb1p = |sb0p| - sh
    sb1pAbs = sb1p[wx-1] ? -sb1p[wx-2:0] : sb1p[wx-2:0];
    sb2p    = {3'b000, shFac[ws-1:1]} - sb1pAbs;          // sb2p = |sb2p| - sh>>1
  
    // Round + shift >>2
    sb0r    = sb0p + 2;
    sb1r    = sb1p + 2;
    sb2r    = sb2p + 2;
    
    sb0s    = sb0r[wx-1:2];
    sb1s    = sb1r[wx-1:2];
    sb2s    = sb2r[wx-1:2];
        
  end

  // Saturation of output soft bits
  sat sat_sb0B(sb0B, sb0Bs);
  defparam sat_sb0B.x_w = wx-4;
  defparam sat_sb0B.y_w = wbb;

  sat sat_sb0(sb0, sb0s);
  defparam sat_sb0.x_w = wx-2;
  defparam sat_sb0.y_w = wb;

  sat sat_sb1(sb1, sb1s);
  defparam sat_sb1.x_w = wx-2;
  defparam sat_sb1.y_w = wb;

  sat sat_sb2(sb2, sb2s);
  defparam sat_sb2.x_w = wx-2;
  defparam sat_sb2.y_w = wb;

  // --------------------------------------------------------------------------------------
  // Assemble output -> to buf1_22 input
  // assign sb = {sb2, sb1, sb0, sb0B};
  assign buf1_22_x = {sb2, sb1, sb0, sb0B};
  
endmodule
//=========================================================================================

