//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   qamMod
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module qamMod(y, y_or, y_ff, y_we, 
              x, x_ir, x_fe, x_re, 
              irqHdrDone, 
              run1, 
              regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter y_w               = 0;
  parameter x_w               = 0;

  parameter irqHdrDone_w      = 0;
  parameter run1_w            = 0;

  parameter mode1_w           = 0;
  parameter mode1_r           = 0;
  parameter mode1_s           = 0;
  parameter R_mode1           = 0;

  parameter w                 = 0;
  parameter bpskL1            = 0;
  parameter qam4L1            = 0;
  parameter qam16L1           = 0;
  parameter qam16L3           = 0;
  parameter qam64L1           = 0;
  parameter qam64L3           = 0;
  parameter qam64L5           = 0;
  parameter qam64L7           = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output            [y_w-1:0]  y;
  output                       y_or;
  input                        y_ff;
  output                       y_we;
  assign                       y_we = y_or & ~y_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  output   [irqHdrDone_w-1:0]  irqHdrDone;
  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire          [mode1_w-1:0]  mode1_q;
  wire          [mode1_w-1:0]  mode1_d;
  wire                         mode1_weint;
  assign                       mode1_weint = 0;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg mode1(mode1_q, regBus, mode1_d, mode1_weint, regWe, regRe, clk, reset);
  defparam mode1.w    = mode1_w;
  defparam mode1.rval = mode1_r;
  defparam mode1.sgn  = mode1_s;
  defparam mode1.adr  = R_mode1;

  // --------------------------------------------------------------------------------------
  // Included instances
  `include "buf1_20.v"

  // --------------------------------------------------------------------------------------
  // Define connections
  assign y                         = buf1_20_y;
  assign y_or                      = buf1_20_y_or;
  assign buf1_20_y_ff              = y_ff;


  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | buf1_20_regWe;
  assign regReOut = 0 | buf1_20_regRe;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------
  // Internal registers and wires
  reg         [5:0] cntBits;                  // Bit counter for PLCP header
  reg         [1:0] xBuf;                     // Input buffer for bits x
  reg         [2:0] xAll;                     // Input buffer for bits x + current x
  reg         [2:0] cntInMaxRe, cntInMaxIm;   // Input counter needed to set Re/Im part of output
  reg         [2:0] cntIn, cntIn_nxt;         // Input counter, next state
  reg       [w-1:0] y0, y1, y2, y3, yN;       // Intermediate values for BPSK, QPSK, 16-QAM, 64-QAM
  reg       [w-1:0] yReOut, yImOut;           // Next value for yRe/yIm
  reg       [w-1:0] yReOutBuf;                // Buffer for next yRe
  
  wire   [y_w-1:0]  yP;                       // Preliminary output    
  wire              yP_we, yP_or, yP_ff;      // before buffering      

  // --------------------------------------------------------------------------
  // Copy this autogenerated stuff to avoid non-causal use of yP
  assign buf1_20_x                 = yP;
  assign buf1_20_x_fe              = ~yP_or;
  assign yP_ff                     = ~buf1_20_x_ir;

  // --------------------------------------------------------------------------
  // Input/output control
  assign x_ir   = ((run1==1 & cntBits!=0) | run1==2) & (yP_ff==0 | cntIn!=cntInMaxIm); 
  assign yP_or  = ((run1==1 & cntBits!=0) | run1==2) & ( x_fe==0 & cntIn==cntInMaxIm);
  assign yP_we = yP_or & ~yP_ff;
  
  // Output buffer control
  assign buf1_20_run1 = run1!=0;
  
  // Set IRQ strobe
  assign irqHdrDone = (run1==1 & cntBits==1 & x_re);
  
  // --------------------------------------------------------------------------
  // Writing address calculation, create interleaving pattern
  always @(*) begin

    // Set number of bits needed (-1) for writing the real/imag part
    case (mode1_q)
      0: cntInMaxRe = 0;
      1: cntInMaxRe = 0;
      2: cntInMaxRe = 1;
      3: cntInMaxRe = 2;
    endcase
    case (mode1_q)
      0: cntInMaxIm = 0;
      1: cntInMaxIm = 1;
      2: cntInMaxIm = 3;
      3: cntInMaxIm = 5;
    endcase

    // Input counter counts 0..cntInMaxIm
    if (cntIn==cntInMaxIm)
      cntIn_nxt = 0;
    else
      cntIn_nxt = cntIn+1;

    // Set xAll = current and 2 delayed input bits
    xAll = {xBuf, x};

    // LUT for BPSK, QPSK, 16-QAM, 64-QAM
    case (xAll[0])
      1'b0:   y0  = -bpskL1;
      1'b1:   y0  =  bpskL1;
    endcase
    case (xAll[0])
      1'b0:   y1  = -qam4L1;
      1'b1:   y1  =  qam4L1;
    endcase
    case (xAll[1:0])
      2'b00:  y2  = -qam16L3;
      2'b01:  y2  = -qam16L1;
      2'b11:  y2  =  qam16L1;
      2'b10:  y2  =  qam16L3;
    endcase
    case (xAll[2:0])
      3'b000: y3  = -qam64L7;
      3'b001: y3  = -qam64L5;
      3'b011: y3  = -qam64L3;
      3'b010: y3  = -qam64L1;
      3'b110: y3  =  qam64L1;
      3'b111: y3  =  qam64L3;
      3'b101: y3  =  qam64L5;
      3'b100: y3  =  qam64L7;
    endcase

    // Select next constellation point according to mode
    case (mode1_q)
      0: yN = y0;
      1: yN = y1;
      2: yN = y2;
      3: yN = y3;
    endcase
    
    // Select next constellation point according to mode
    yReOut = yN;                  // Use mapped point for real part
    yImOut = mode1_q==0 ? 0 : yN; // Special treatment for BPSK imag part
      
  end
      
  // --------------------------------------------------------------------------
  // Main FSM
  always @(posedge clk) begin
    
    // Reset
    if (run1==0) begin
      cntIn   <= 0;
      cntBits <= 48;
  
    // Normal operation
    end else begin
      
      // Update state upon reading input data
      if (x_re) begin

        // Shift input, count input bits
        xBuf  <= {xBuf[0], x};
        cntIn <= cntIn_nxt;
        
        // Buffer real part
        if (cntIn==cntInMaxRe) begin  
          yReOutBuf <= yReOut;
        end

        // Count down bits in PLCP header mode
        if (run1==1) begin 
          cntBits <= cntBits-1;
        end
        
      end      
  
    end
  end
  
  // --------------------------------------------------------------------------
  // Assign output y - use yRe directly for BPSK, otherwise buffered value
  assign yP = {yImOut, mode1_q==0 ? yReOut : yReOutBuf};     
 
  // --------------------------------------------------------------------------

endmodule
//=========================================================================================

