//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   ppdu11b
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module ppdu11b(y, y_or, y_ff, y_we, 
               x, x_ir, x_fe, x_re, 
               run1, 
               regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter y_w               = 0;
  parameter x_w               = 0;

  parameter run1_w            = 0;

  parameter header_w          = 0;
  parameter header_r          = 0;
  parameter header_s          = 0;
  parameter R_header          = 0;
  parameter sfd_w             = 0;
  parameter sfd_r             = 0;
  parameter sfd_s             = 0;
  parameter R_sfd             = 0;
  parameter sigser_w          = 0;
  parameter sigser_r          = 0;
  parameter sigser_s          = 0;
  parameter R_sigser          = 0;
  parameter len_w             = 0;
  parameter len_r             = 0;
  parameter len_s             = 0;
  parameter R_len             = 0;
  parameter crc_w             = 0;
  parameter crc_r             = 0;
  parameter crc_s             = 0;
  parameter R_crc             = 0;
  parameter numBits_w         = 0;
  parameter numBits_r         = 0;
  parameter numBits_s         = 0;
  parameter R_numBits         = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output            [y_w-1:0]  y;
  output                       y_or;
  input                        y_ff;
  output                       y_we;
  assign                       y_we = y_or & ~y_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire         [header_w-1:0]  header_q;
  wire         [header_w-1:0]  header_d;
  wire                         header_weint;
  assign                       header_weint = 0;

  wire            [sfd_w-1:0]  sfd_q;
  wire            [sfd_w-1:0]  sfd_d;
  wire                         sfd_weint;
  assign                       sfd_weint = 0;

  wire         [sigser_w-1:0]  sigser_q;
  wire         [sigser_w-1:0]  sigser_d;
  wire                         sigser_weint;
  assign                       sigser_weint = 0;

  wire            [len_w-1:0]  len_q;
  wire            [len_w-1:0]  len_d;
  wire                         len_weint;
  assign                       len_weint = 0;

  wire            [crc_w-1:0]  crc_q;
  wire            [crc_w-1:0]  crc_d;
  wire                         crc_weint;
  assign                       crc_weint = 0;

  wire        [numBits_w-1:0]  numBits_q;
  wire        [numBits_w-1:0]  numBits_d;
  wire                         numBits_weint;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg header(header_q, regBus, header_d, header_weint, regWe, regRe, clk, reset);
  defparam header.w    = header_w;
  defparam header.rval = header_r;
  defparam header.sgn  = header_s;
  defparam header.adr  = R_header;

  ereg sfd(sfd_q, regBus, sfd_d, sfd_weint, regWe, regRe, clk, reset);
  defparam sfd.w    = sfd_w;
  defparam sfd.rval = sfd_r;
  defparam sfd.sgn  = sfd_s;
  defparam sfd.adr  = R_sfd;

  ereg sigser(sigser_q, regBus, sigser_d, sigser_weint, regWe, regRe, clk, reset);
  defparam sigser.w    = sigser_w;
  defparam sigser.rval = sigser_r;
  defparam sigser.sgn  = sigser_s;
  defparam sigser.adr  = R_sigser;

  ereg len(len_q, regBus, len_d, len_weint, regWe, regRe, clk, reset);
  defparam len.w    = len_w;
  defparam len.rval = len_r;
  defparam len.sgn  = len_s;
  defparam len.adr  = R_len;

  ereg crc(crc_q, regBus, crc_d, crc_weint, regWe, regRe, clk, reset);
  defparam crc.w    = crc_w;
  defparam crc.rval = crc_r;
  defparam crc.sgn  = crc_s;
  defparam crc.adr  = R_crc;

  ereg numBits(numBits_q, regBus, numBits_d, numBits_weint, regWe, regRe, clk, reset);
  defparam numBits.w    = numBits_w;
  defparam numBits.rval = numBits_r;
  defparam numBits.sgn  = numBits_s;
  defparam numBits.adr  = R_numBits;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================
  
  // -----------------------------------------------------------------------------------------
  // Internal registers and variables  
  reg                     state, state_nxt;                 // state
  reg              [7:0]  cnt, cnt_nxt;                     // internal state counter
  reg             [15:0]  sreg, sreg_nxt;                   // shift register
  
  wire                    runIt;                            // continue state machine
  wire                    dr_nxt;                           // dr = data ready at the
  reg                     dr;                               //      output y



  // -------------------------------------------------------------------------------------
  // States for reading and writing

  // Run the state machine if 
  //  - run1==1
  //  - data is not ready or currently read
  //  - state == 0 or data is available at input
  assign runIt  = run1 & (~dr | y_we) & (state==0 | ~x_fe);

  // Data will be ready in the next cycle if
  //  - run1==1
  //  - the state machine is run or (data is ready and not read currently)
  assign dr_nxt = run1 & (runIt | (dr & ~y_we));

  // Read data from the input if
  //  - state == 1
  //  - output not ready, or written
  assign x_ir   = (~dr | y_we) & state==1 & numBits_q!=0; 

  // Write data to the output if 
  // - data is ready 
  assign y_or   = dr;

  // Output the last bit of the shift register
  assign y      = sreg[0];

  // Update remaining number of bits when input is read
  assign numBits_d      = numBits_q-1;
  assign numBits_weint  = x_re;

  // -----------------------------------------------------------------------------------------
  // Main loop
  always @(*) begin

    // --------------------------------------
    // Use old states if nothing else defined
    cnt_nxt     = cnt;
    sreg_nxt    = sreg;
    state_nxt   = state;
    
    // --------------------------------------
    // Reset state if run1==0
    if (run1==0) begin
      state_nxt     = 0;
      if (header_q) begin
        sreg_nxt  = 16'hFFFF;                       // transmit all ones
        cnt_nxt   = 192;
      end else begin
        sreg_nxt  = 16'h0000;                       // transmit all zeros
        cnt_nxt   = 120;
      end

    // --------------------------------------
    // Normal operation if run1==1
    end else begin      
      if (runIt) begin

        //state 0: transmit sync, sfd, sig/serv, len, crc
        if (state==0) begin

          cnt_nxt = cnt-1;                                  // cnt down
            
          case (cnt)                                        // set the shift reg depending
            64 : sreg_nxt = sfd_q;                          // on the counter state
            48 : sreg_nxt = sigser_q;
            32 : sreg_nxt = len_q;
            16 : sreg_nxt = crc_q;
            default: sreg_nxt = {sreg[0], sreg[15:1]};      // rotate right
          endcase

          if (cnt==1) begin                                 // goto  psdu transmit
            state_nxt = 1;
          end
    
        // state 1: transmit PSDU data
        end else begin
          sreg_nxt[0] = x;
        end   

      end //if (runIt)
    end //normal operation
  end //always
  
  // --------------------------------------
  // FSM-feedback
  always @(posedge clk) begin
    
    cnt       <= cnt_nxt;                                   // simply the feedback here
    sreg      <= sreg_nxt;
    state     <= state_nxt;
    dr        <= dr_nxt;

  end //always @(posedge clk)

endmodule
//========================================================================================
