//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   lsCcf
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module lsCcf(yAbs, yAbs_or, yAbs_ff, yAbs_we, 
             yFft, yFft_or, yFft_ff, yFft_we, 
             yC, yC_or, yC_ff, yC_we, 
             x, x_ir, x_fe, x_re, 
             xFft, xFft_ir, xFft_fe, xFft_re, 
             irqFteDone, 
             irqCtfDone, 
             run1, 
             regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Include global definitions
  `include "def_Const_wlan.v"

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter yAbs_w            = 0;
  parameter yFft_w            = 0;
  parameter yC_w              = 0;
  parameter x_w               = 0;
  parameter xFft_w            = 0;

  parameter irqFteDone_w      = 0;
  parameter irqCtfDone_w      = 0;
  parameter run1_w            = 0;

  parameter peakPos_w         = 0;
  parameter peakPos_r         = 0;
  parameter peakPos_s         = 0;
  parameter R_peakPos         = 0;
  parameter peakVal_w         = 0;
  parameter peakVal_r         = 0;
  parameter peakVal_s         = 0;
  parameter R_peakVal         = 0;

  parameter wx                = 0;
  parameter wa                = 0;
  parameter wc                = 0;
  parameter wf                = 0;
  parameter StFft             = 0;
  parameter StMul             = 0;
  parameter StMax             = 0;
  parameter StHalt            = 0;
  parameter ModeReset         = 0;
  parameter ModeFte           = 0;
  parameter ModeChEst         = 0;
  parameter ModeChCor         = 0;
  parameter SyncOff           = 0;
  parameter SyncFte           = 0;
  parameter SyncFfe           = 0;
  parameter SyncAmpEst        = 0;
  parameter SyncChEst         = 0;
  parameter SyncChCor         = 0;

  parameter state_w           = 0;
  parameter state_n           = 0;
  parameter state_m           = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output         [yAbs_w-1:0]  yAbs;
  output                       yAbs_or;
  input                        yAbs_ff;
  output                       yAbs_we;
  assign                       yAbs_we = yAbs_or & ~yAbs_ff;

  output         [yFft_w-1:0]  yFft;
  output                       yFft_or;
  input                        yFft_ff;
  output                       yFft_we;
  assign                       yFft_we = yFft_or & ~yFft_ff;

  output           [yC_w-1:0]  yC;
  output                       yC_or;
  input                        yC_ff;
  output                       yC_we;
  assign                       yC_we = yC_or & ~yC_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  input          [xFft_w-1:0]  xFft;
  output                       xFft_ir;
  input                        xFft_fe;
  output                       xFft_re;
  assign                       xFft_re = xFft_ir & ~xFft_fe;

  output   [irqFteDone_w-1:0]  irqFteDone;
  output   [irqCtfDone_w-1:0]  irqCtfDone;
  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire        [peakPos_w-1:0]  peakPos_q;
  wire        [peakPos_w-1:0]  peakPos_d;
  wire                         peakPos_weint;

  wire        [peakVal_w-1:0]  peakVal_q;
  wire        [peakVal_w-1:0]  peakVal_d;
  wire                         peakVal_weint;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg peakPos(peakPos_q, regBus, peakPos_d, peakPos_weint, regWe, regRe, clk, reset);
  defparam peakPos.w    = peakPos_w;
  defparam peakPos.rval = peakPos_r;
  defparam peakPos.sgn  = peakPos_s;
  defparam peakPos.adr  = R_peakPos;

  ereg peakVal(peakVal_q, regBus, peakVal_d, peakVal_weint, regWe, regRe, clk, reset);
  defparam peakVal.w    = peakVal_w;
  defparam peakVal.rval = peakVal_r;
  defparam peakVal.sgn  = peakVal_s;
  defparam peakVal.adr  = R_peakVal;

  // --------------------------------------------------------------------------------------
  // Internal status registers
  reg           [state_w-1:0]  state;

  // --------------------------------------------------------------------------------------
  // Included instances
  `include "addShiftPass_0.v"
  `include "lsMul_0.v"
  `include "s2To1Sel_20.v"
  `include "s1To2Sel_20a.v"
  `include "s1To2Sel_20b.v"
  `include "buf0_20.v"

  // --------------------------------------------------------------------------------------
  // Define connections
  assign addShiftPass_0_x          = x;
  assign addShiftPass_0_x_fe       = x_fe;
  assign x_ir                      = addShiftPass_0_x_ir;

  assign buf0_20_x                 = xFft;
  assign buf0_20_x_fe              = xFft_fe;
  assign xFft_ir                   = buf0_20_x_ir;

  assign s1To2Sel_20a_x            = buf0_20_y;
  assign s1To2Sel_20a_x_fe         = ~buf0_20_y_or;
  assign buf0_20_y_ff              = ~s1To2Sel_20a_x_ir;

  assign lsMul_0_x                 = s1To2Sel_20a_y0;
  assign lsMul_0_x_fe              = ~s1To2Sel_20a_y0_or;
  assign s1To2Sel_20a_y0_ff        = ~lsMul_0_x_ir;

  assign s1To2Sel_20b_x            = lsMul_0_y;
  assign s1To2Sel_20b_x_fe         = ~lsMul_0_y_or;
  assign lsMul_0_y_ff              = ~s1To2Sel_20b_x_ir;

  assign s2To1Sel_20_x0            = addShiftPass_0_y;
  assign s2To1Sel_20_x0_fe         = ~addShiftPass_0_y_or;
  assign addShiftPass_0_y_ff       = ~s2To1Sel_20_x0_ir;

  assign s2To1Sel_20_x1            = s1To2Sel_20b_y1;
  assign s2To1Sel_20_x1_fe         = ~s1To2Sel_20b_y1_or;
  assign s1To2Sel_20b_y1_ff        = ~s2To1Sel_20_x1_ir;

  assign yFft                      = s2To1Sel_20_y;
  assign yFft_or                   = s2To1Sel_20_y_or;
  assign s2To1Sel_20_y_ff          = yFft_ff;

  assign yC                        = s1To2Sel_20b_y0;
  assign yC_or                     = s1To2Sel_20b_y0_or;
  assign s1To2Sel_20b_y0_ff        = yC_ff;


  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | addShiftPass_0_regWe | lsMul_0_regWe | s2To1Sel_20_regWe
                      | s1To2Sel_20a_regWe | s1To2Sel_20b_regWe | buf0_20_regWe;
  assign regReOut = 0 | addShiftPass_0_regRe | lsMul_0_regRe | s2To1Sel_20_regRe
                      | s1To2Sel_20a_regRe | s1To2Sel_20b_regRe | buf0_20_regRe;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Register definitions
  reg    [wf-1:0]  xFftRe, xFftIm;
  reg    [wf-1:0]  absXFft;
  reg      [wf:0]  maxDetect;
  reg              stopFftIn; 
  
  reg       [5:0]  cnt;
  reg      [63:0]  lsPat0, lsPat1;
    
  wire      [1:0]  runMode;  // 0=lsCcf, 1=ChEst, 2=ChCor
  
  // --------------------------------------------------------------------------------------
  // Get run mode, 0 = off, 1=CTE, 2=ChEst, 3=ChCor
  assign runMode = run1==WlanSyncOff_C   ? ModeReset : 
                   run1==WlanSyncChEst_C ? ModeChEst : 
                   run1==WlanSyncChCor_C ? ModeChCor : 
                                           ModeFte;
  // Block FFT input via buf0
  assign buf0_20_run1 = ~stopFftIn;

  // Assign run1 inputs of submodules
  assign addShiftPass_0_run1  = runMode==ModeChEst & state==StMul ? ModeChCor : runMode;
  assign lsMul_0_run1         = runMode;
  
  // Assign switches
  assign s2To1Sel_20_sel      = runMode==ModeFte & state==StMul;
  assign s1To2Sel_20a_sel     = runMode==ModeFte & state==StMax;
  assign s1To2Sel_20a_y1_ff   = 0;
  assign s1To2Sel_20b_sel     = (runMode == ModeFte);

  // Assign interrupt - at transition to state=StDone
  assign irqFteDone   = state==StMax & cnt==63 & s1To2Sel_20a_y1_we;
  assign irqCtfDone   = state==StMul & cnt==59 & yC_we & runMode==ModeChEst; 
 
  // Assign input of the peak position and value
  assign peakPos_d     = run1==0 ? 0 : cnt-16;
  assign peakVal_d     = run1==0 ? 0 : absXFft;
  assign peakPos_weint = run1==0 | (state==StMax & s1To2Sel_20a_y1_we & maxDetect[wf]==1);
  assign peakVal_weint = run1==0 | (state==StMax & s1To2Sel_20a_y1_we & maxDetect[wf]==1);

  // --------------------------------------------------------------------------------------
  // Signal processing behaviour for maximum search
  always @(*) begin

    // Split input from FFT into real and imag part
    xFftRe = s1To2Sel_20a_y1[  wf-1: 0];
    xFftIm = s1To2Sel_20a_y1[2*wf-1:wf];
 
    // Calculation of the final output |x| = |xRe|+|xIm| (CCF only)
    absXFft = (xFftRe[wf-1] ? -xFftRe : xFftRe) + 
              (xFftIm[wf-1] ? -xFftIm : xFftIm);
    maxDetect = {1'b0, peakVal_q} - {1'b0, absXFft};   
    
  end

  // Debugging: output absXfft
  assign yAbs     = absXFft;
  assign yAbs_or  = s1To2Sel_20a_y1_or;

  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin
    
    // Reset
    if (run1==0) begin
      cnt       <= 0;
      stopFftIn <= 0;
      state     <= StFft;
                       
    // Normal operation
    end else begin
      
      // Counter update
      if ( (runMode==ModeFte   & ((state!=StMax & yFft_we) | (state==StMax & xFft_re))) |
           (runMode==ModeChEst & ((state==StFft & yFft_we) | (state==StMul & xFft_re))) ) begin
        cnt <= cnt+1;
      end
            
      // FSM update - depends on run mode
      if (runMode==ModeFte) begin
        
        if (state==StFft) begin
          if (cnt==63 & yFft_we) begin
            state <= StMul;
          end
        end else
        if (state==StMul) begin
          if (cnt==63 & yFft_we) begin
            state <= StMax;
          end
        end else
        if (state==StMax) begin
          if (cnt==63 & xFft_re) begin
            state <= StFft;
          end
        end
      end else
      
      if (runMode==ModeChEst) begin
        if (state==StFft) begin
          if (cnt==63 & yFft_we) begin
            state <= StMul;
          end
        end
      end

      // FFT input stop after channel estimation
      if (runMode==ModeChEst & state==StMul & cnt==63 & xFft_re) begin
        stopFftIn <= 1;
      end else
      if (runMode==ModeChCor) begin
        stopFftIn <= 0;
      end
                    
    end // if (run1_q==0) ... else
  end // always

endmodule
//=========================================================================================

