//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   deIntlv
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module deIntlv(y, y_or, y_ff, y_we, 
               x, x_ir, x_fe, x_re, 
               run1, 
               regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter y_w               = 0;
  parameter x_w               = 0;

  parameter run1_w            = 0;

  parameter mode1_w           = 0;
  parameter mode1_r           = 0;
  parameter mode1_s           = 0;
  parameter R_mode1           = 0;

  parameter wy                = 0;
  parameter wb                = 0;

  parameter cntBlock_w        = 0;
  parameter cntBlock_n        = 0;
  parameter cntBlock_m        = 0;

  parameter m0_w              = 0;
  parameter m0_n              = 0;
  parameter m0_m              = 0;
  parameter m1_w              = 0;
  parameter m1_n              = 0;
  parameter m1_m              = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output            [y_w-1:0]  y;
  output                       y_or;
  input                        y_ff;
  output                       y_we;
  assign                       y_we = y_or & ~y_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire          [mode1_w-1:0]  mode1_q;
  wire          [mode1_w-1:0]  mode1_d;
  wire                         mode1_weint;
  assign                       mode1_weint = 0;

  // Wires for internal memories
  wire             [m0_w-1:0]  m0_q;
  wire             [m0_w-1:0]  m0_d;
  wire             [m0_m-1:0]  m0_rdAdr;
  wire             [m0_m-1:0]  m0_wrAdr;
  wire                         m0_we;
  wire             [m1_w-1:0]  m1_q;
  wire             [m1_w-1:0]  m1_d;
  wire             [m1_m-1:0]  m1_rdAdr;
  wire             [m1_m-1:0]  m1_wrAdr;
  wire                         m1_we;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg mode1(mode1_q, regBus, mode1_d, mode1_weint, regWe, regRe, clk, reset);
  defparam mode1.w    = mode1_w;
  defparam mode1.rval = mode1_r;
  defparam mode1.sgn  = mode1_s;
  defparam mode1.adr  = R_mode1;

  // --------------------------------------------------------------------------------------
  // Internal status registers
  reg        [cntBlock_w-1:0]  cntBlock;

  // --------------------------------------------------------------------------------------
  // Internal dual ported memories
  dpRam1 m0(m0_q, m0_d, m0_rdAdr, m0_wrAdr, m0_we, clk);
  defparam m0.w    = m0_w;
  defparam m0.n    = m0_n;
  defparam m0.m    = m0_m;

  dpRam1 m1(m1_q, m1_d, m1_rdAdr, m1_wrAdr, m1_we, clk);
  defparam m1.w    = m1_w;
  defparam m1.n    = m1_n;
  defparam m1.m    = m1_m;

  // --------------------------------------------------------------------------------------
  // Included instances
  `include "buf2_8.v"

  // --------------------------------------------------------------------------------------
  // Define connections
  assign y                         = buf2_8_y;
  assign y_or                      = buf2_8_y_or;
  assign buf2_8_y_ff               = y_ff;


  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | buf2_8_regWe;
  assign regReOut = 0 | buf2_8_regRe;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------
  // Define states, consider the 2 memories a 2 entry FIFO, copy from fifoA
  reg    [4:0]  cntIn,  cntInNxt;              // Input data counter 0..31, carriers (re/im)
  reg    [5:0]  cntOut, cntOutNxt, cntOutMax;  // Output counter 0..6*8-1 maximum
  reg    [5:0]  cntX;
  reg    [5:0]  cntY;
  wire          lastOut, firstOut;
  
  // Define registers and wires for interleaving and memory write
  reg              [1:0]  idxAdr0, idxAdr0Nxt;
  reg              [1:0]  idxAdr1, idxAdr1Nxt;
  reg                     idxReIm, idxReImNxt;
  wire             [3:0]  rdAdr;
  
  wire          [wy-1:0]  sbd0xb, sbd1xb;
  wire          [wb-1:0]  sbd0x0, sbd1x0;
  wire          [wb-1:0]  sbd0x1, sbd1x1;
  wire          [wb-1:0]  sbd0x2, sbd1x2;
  wire  signed  [wb-1:0]  sbd0,   sbd1;
  wire  signed  [wb-1:0]  sbSh0,  sbSh1;
  wire          [wy-1:0]  sbSat0, sbSat1;
  wire          [wy-1:0]  sbY0,   sbY1;
  
  wire        [2*wy-1:0]  y0;             // Intermediate output
  wire                    y0_or, y0_we;   // before buffering
  
  // --------------------------------------------------------------------------
  // Define input and output control
  assign x_ir = run1!=0 && cntX!=32;
  
  // --------------------------------------------------------------------------
   // Debug only
  wire  [wy-1:0] x0;
  wire  [wb-1:0] x1;
  wire  [wb-1:0] x2;
  wire  [wb-1:0] x3;
  
  assign x0 = x[     wy-1:      0];
  assign x1 = x[1*wb+wy-1:0*wb+wy]; 
  assign x2 = x[2*wb+wy-1:1*wb+wy]; 
  assign x3 = x[3*wb+wy-1:2*wb+wy]; 
  
  // --------------------------------------------------------------------------
  // FIFO counter FSM
  assign firstOut = cntOut[5:3]==0;
  assign lastOut  = (mode1_q==0) ? (cntOut[5:3]==0) :
                    (mode1_q==1) ? (cntOut[5:3]==1) :
                    (mode1_q==2) ? (cntOut[5:3]==3) :
                                   (cntOut[5:3]==5);
  always @(posedge clk) begin
      
    if (run1==0) begin
      cntX <= 0;
      cntY <= 0;
    end else
    
    // Normal processing
    begin
      
      // Input counter
      if (x_re && y0_we && lastOut) begin
        cntX <= cntX - 3;
      end else
      if (x_re) begin
        cntX <= cntX + 1;
      end else
      if (y0_we && lastOut) begin
        cntX <= cntX - 4;
      end
      
      // Output counter
      if (x_re && y0_we && firstOut) begin
        cntY <= cntY - 3;
      end else
      if (x_re) begin
        cntY <= cntY + 1;
      end else
      if (y0_we && firstOut) begin
        cntY <= cntY - 4;
      end
            
    end
  
  end
  
  
  // --------------------------------------------------------------------------
  // Input FSM
  always @(posedge clk) begin
 
    // ---------------------------------
    // Reset
    if (run1==0) begin
      cntIn    <= 0;
    end else

    // Normal operation
    begin
      if (x_re) begin
        if (cntIn != 31) begin
          cntIn <= cntIn + 1;
        end else begin
          cntIn <= 0;
        end
      end
    end    
  end // always  

  // --------------------------------------------------------------------------
  // Write input to memories
  assign m0_d     = x;
  assign m0_wrAdr = {cntIn[4:2], cntIn[0]};
  assign m0_we    = x_re && (cntIn[1]==0);
  
  assign m1_d     = x;
  assign m1_wrAdr = {cntIn[4:2], cntIn[0]};
  assign m1_we    = x_re && (cntIn[1]==1);

  // --------------------------------------------------------------------------
  // Output FSM
  always @(posedge clk) begin
    
    if (run1==0) begin
      cntOut   <= 0;
      idxAdr0  <= 0;
      idxAdr1  <= 0;
      idxReIm  <= 0;
      cntBlock <= 0;
    end else
  
    // ---------------------------------
    // Normal operation
    begin
      
      // Update output counters
      if (y0_we) begin
        cntOut  <= cntOutNxt;
        idxAdr0 <= idxAdr0Nxt;
        idxAdr1 <= idxAdr1Nxt;
        idxReIm <= idxReImNxt;
        if (run1==1 && cntOut==7) begin
          cntBlock <= cntBlock+1;
        end
      end
      
      // Proper init of idxAdr
      if (run1==1 && cntBlock==3) begin
        idxAdr1 <= mode1_q-1;
      end
      
    end
    
  end // always
  
  // --------------------------------------------------------------------------
  // Output read index calculation
  always @(*) begin
      
    // Default setting - keep old values
    idxAdr0Nxt = idxAdr0;
    idxAdr1Nxt = idxAdr1;
    idxReImNxt = idxReIm;
    
    // Set counter maximum
    if (mode1_q==0) begin
      cntOutMax = 7;
    end else begin
      cntOutMax = { {mode1_q, 1'b0}-3'b001, 3'b111};
    end
    
    // Update counter
    if (cntOut==cntOutMax) begin
      cntOutNxt = 0;
    end else begin
      cntOutNxt = cntOut + 1;
    end  
    
    if (((cntOut==(cntOutMax>>1)) || (cntOut==cntOutMax)) && (mode1_q!=0)) begin
      idxReImNxt = ~idxReIm;
    end  
    
    // QPSK
    if (mode1_q==1) begin
      idxAdr0Nxt = 0;
      idxAdr1Nxt = 0;
    end else
    
    // 16-QAM
    if (mode1_q==2) begin
      if (cntOut[2:0]==7) begin
        idxAdr0Nxt = idxAdr0 ^ 2'b01;
        idxAdr1Nxt = idxAdr1 ^ 2'b01;
      end
    end else
    
    // 64-QAM
    if (mode1_q==3) begin
      if (cntOut[2:0]==7) begin
        idxAdr0Nxt = idxAdr0;
      end else
      if (idxAdr0==2) begin
        idxAdr0Nxt = 0;
      end else
      begin
        idxAdr0Nxt = idxAdr0 + 1;
      end

      if (cntOut[2:0]==7) begin
        idxAdr1Nxt = idxAdr1;
      end else
      if (idxAdr1==2) begin
        idxAdr1Nxt = 0;
      end else
      begin
        idxAdr1Nxt = idxAdr1 + 1;
      end
    end

  end
  
  // --------------------------------------------------------------------------
  // Get read addresses
  assign rdAdr    = y0_we ? {cntOutNxt[2:0], idxReImNxt} : {cntOut[2:0], idxReIm};

  assign m0_rdAdr = rdAdr;
  assign sbd0xb   = m0_q[wy-1:0];
  assign sbd0x0   = m0_q[1*wb-1+wy:0*wb+wy];
  assign sbd0x1   = m0_q[2*wb-1+wy:1*wb+wy];
  assign sbd0x2   = m0_q[3*wb-1+wy:2*wb+wy];

  assign m1_rdAdr = rdAdr;
  assign sbd1xb   = m1_q[wy-1:0];
  assign sbd1x0   = m1_q[1*wb-1+wy:0*wb+wy];
  assign sbd1x1   = m1_q[2*wb-1+wy:1*wb+wy];
  assign sbd1x2   = m1_q[3*wb-1+wy:2*wb+wy];

  // Select memory output from deinterleaved soft bits
  assign sbd0 = (idxAdr0 == 0) ? sbd0x0 :
                (idxAdr0 == 1) ? sbd0x1 :
                (idxAdr0 == 2) ? sbd0x2 :
                0;
  assign sbd1 = (idxAdr1 == 0) ? sbd1x0 :
                (idxAdr1 == 1) ? sbd1x1 :
                (idxAdr1 == 2) ? sbd1x2 :
                0;
  
  // Shift and saturate  
  assign sbSh0 = (mode1_q==1) ? (sbd0 >>> 2) :
                 (mode1_q==2) ? (sbd0 >>> 1) :
                                 sbd0;
  assign sbSh1 = (mode1_q==1) ? (sbd1 >>> 2) :
                 (mode1_q==2) ? (sbd1 >>> 1) :
                                 sbd1;

  sat sat_0(sbSat0, sbSh0);
  defparam sat_0.x_w = wb;
  defparam sat_0.y_w = wy;
  
  sat sat_1(sbSat1, sbSh1);
  defparam sat_1.x_w = wb;
  defparam sat_1.y_w = wy;

  // Multiplex output with directy BPSK stream
  assign sbY0 = (mode1_q==0) ? sbd0xb : sbSat0;
  assign sbY1 = (mode1_q==0) ? sbd1xb : sbSat1;
  
  // Combine output
  assign y0    = {sbY1, sbY0};
  assign y0_or = ((run1==1 && cntBlock!=3) || run1==2 || run1==3) &&
                 (cntY[5:2]!=0 || ~firstOut); 
  
  // --------------------------------------------------------------------------  
  // Register output using buffer
  assign buf2_8_run1  = (run1!=0);
  assign buf2_8_x     = y0;
  assign buf2_8_x_fe  = ~y0_or;
  assign y0_we        = buf2_8_x_re;

endmodule
//=========================================================================================

