//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   cxMulAvg
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module cxMulAvg(ySh, ySh_or, ySh_ff, ySh_we, 
                yC, yC_or, yC_ff, yC_we, 
                x, x_ir, x_fe, x_re, 
                irqDone, 
                run1, 
                regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters
  parameter ySh_w             = 0;
  parameter yC_w              = 0;
  parameter x_w               = 0;

  parameter irqDone_w         = 0;
  parameter run1_w            = 0;

  parameter numSmpl_w         = 0;
  parameter numSmpl_r         = 0;
  parameter numSmpl_s         = 0;
  parameter R_numSmpl         = 0;
  parameter yReLo_w           = 0;
  parameter yReLo_r           = 0;
  parameter yReLo_s           = 0;
  parameter R_yReLo           = 0;
  parameter yImHi_w           = 0;
  parameter yImHi_r           = 0;
  parameter yImHi_s           = 0;
  parameter R_yImHi           = 0;

  parameter wx                = 0;
  parameter wy                = 0;
  parameter wc                = 0;
  parameter sh1               = 0;
  parameter sh2               = 0;
  parameter sh3               = 0;
  parameter sh4               = 0;
  parameter cxMulModeFfe      = 0;
  parameter cxMulModeAmpEst   = 0;
  parameter cxMulModeChEst    = 0;
  parameter cxMulModeChCor    = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output          [ySh_w-1:0]  ySh;
  output                       ySh_or;
  input                        ySh_ff;
  output                       ySh_we;
  assign                       ySh_we = ySh_or & ~ySh_ff;

  output           [yC_w-1:0]  yC;
  output                       yC_or;
  input                        yC_ff;
  output                       yC_we;
  assign                       yC_we = yC_or & ~yC_ff;

  input             [x_w-1:0]  x;
  output                       x_ir;
  input                        x_fe;
  output                       x_re;
  assign                       x_re = x_ir & ~x_fe;

  output      [irqDone_w-1:0]  irqDone;
  input          [run1_w-1:0]  run1;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire        [numSmpl_w-1:0]  numSmpl_q;
  wire        [numSmpl_w-1:0]  numSmpl_d;
  wire                         numSmpl_weint;
  assign                       numSmpl_weint = 0;

  wire          [yReLo_w-1:0]  yReLo_q;
  wire          [yReLo_w-1:0]  yReLo_d;
  wire                         yReLo_weint;

  wire          [yImHi_w-1:0]  yImHi_q;
  wire          [yImHi_w-1:0]  yImHi_d;
  wire                         yImHi_weint;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg numSmpl(numSmpl_q, regBus, numSmpl_d, numSmpl_weint, regWe, regRe, clk, reset);
  defparam numSmpl.w    = numSmpl_w;
  defparam numSmpl.rval = numSmpl_r;
  defparam numSmpl.sgn  = numSmpl_s;
  defparam numSmpl.adr  = R_numSmpl;

  ereg yReLo(yReLo_q, regBus, yReLo_d, yReLo_weint, regWe, regRe, clk, reset);
  defparam yReLo.w    = yReLo_w;
  defparam yReLo.rval = yReLo_r;
  defparam yReLo.sgn  = yReLo_s;
  defparam yReLo.adr  = R_yReLo;

  ereg yImHi(yImHi_q, regBus, yImHi_d, yImHi_weint, regWe, regRe, clk, reset);
  defparam yImHi.w    = yImHi_w;
  defparam yImHi.rval = yImHi_r;
  defparam yImHi.sgn  = yImHi_s;
  defparam yImHi.adr  = R_yImHi;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Register definitions
  reg                           x_reD, x_reDD;
  reg                           xInFlag;
  reg               [2*wx-1:0]  xD, xIn0, xIn1;
  wire signed         [wx-1:0]  xRe, xIm, xRe0, xIm0, xRe1, xIm1;
  reg  signed         [wx-1:0]  mulAIn0, mulAIn1, mulBIn0, mulBIn1;
  reg  signed       [2*wx-2:0]  mulAOut, mulBOut;
  wire signed       [2*wx-1:0]  mulAddOut;
  reg  signed     [2*wx-1+6:0]  addOut, addIn0, addIn1;
  reg  signed     [2*wx-1+6:0]  accRe, accIm;
  reg                    [7:0]  cnt;
  wire                   [7:0]  cntP1;
  reg                    [1:0]  procCxMul;
  reg                    [1:0]  procCxAdd;
  reg                           yC_orR;
  wire                          runFfe, runAmpEst, runShFac, runChCor, resRdy;
  
  // --------------------------------------------------------------------------------------
  // Assign input enable and run flag
  assign runFfe     = run1==cxMulModeFfe;
  assign runAmpEst  = run1==cxMulModeAmpEst;
  assign runShFac   = run1==cxMulModeChEst;
  assign runChCor   = run1==cxMulModeChCor;
  assign x_ir       = runFfe    | 
                      runAmpEst | 
                      runShFac  | 
                      runChCor;

  // Input counter
  assign cntP1 = cnt + 1;
    
  // ---------------------------------------------------------------------------
  // Input buffering, counting and flags
  always @(posedge clk) begin

    // Reset
    if (run1==0) begin
      cnt     <= 0;
      x_reD   <= 0;
      x_reDD  <= 0;
      xInFlag <= 0;
      
    // Normal operation
    end else begin

      // Delayed input flags
      x_reD     <= x_re;
      x_reDD    <= x_reD;
      
      // Toggle input flag
      if (x_re) begin
        xInFlag <= ~xInFlag;
      end
      
      // Buffer input data
      if (x_re) begin
        if (xInFlag==0) begin
          xIn0 <= x;
        end else begin
          xIn1 <= x;
        end
      end
      
      // Input data counter 0..2*numSmpl-1
      if (x_reD) begin
        if (( (runFfe | runChCor) & cntP1[7:1]==numSmpl_q) | 
            (~(runFfe | runChCor) & cntP1[6:0]==numSmpl_q)) begin
          cnt <= 0;
        end else begin
          cnt <= cntP1;
        end
      end
    end
  end
  
  // Split inputs into real/imag part
  assign  xRe   = x[  wx-1: 0];
  assign  xIm   = x[2*wx-1:wx];
  assign  xRe0  = xIn0[  wx-1: 0];
  assign  xIm0  = xIn0[2*wx-1:wx];
  assign  xRe1  = xIn1[  wx-1: 0];
  assign  xIm1  = xIn1[2*wx-1:wx];

  // ---------------------------------------------------------------------------
  // Multiplier input multiplexing
  always @(*) begin
    
    // Multiplier inputs - FFE
    if (runFfe | runChCor) begin
      if (procCxMul==1) begin
        mulAIn0 = xRe0;    // Real part
        mulAIn1 = xRe;
        mulBIn0 = xIm0;
        mulBIn1 = xIm;
      end else 
      begin
        mulAIn0 = xRe0;    // Imag part
        mulAIn1 = xIm1;
        mulBIn0 = xIm0;
        mulBIn1 = -xRe1;
      end
    end else
    
    // Multiplier inputs - AmpEst, ShFac calc
    begin
      mulAIn0 = xRe;
      mulAIn1 = xRe;
      mulBIn0 = xIm;
      mulBIn1 = xIm;
    end    
  end
  
  // Update mul stage
  always @(posedge clk) begin
    if (run1==0) begin
      procCxMul <= 0;
    end else begin
      if (procCxMul==0 & x_re) begin
        procCxMul <= 1;
      end else
      if (procCxMul==1 & x_re) begin
        procCxMul <= 2;
      end else
      if (procCxMul==2 & x_re) begin
        procCxMul <= 1;
      end else
      if (procCxMul==2 & ~x_re) begin
        procCxMul <= 0;
      end
    end
    mulAOut <= mulAIn0 * mulAIn1;
    mulBOut <= mulBIn0 * mulBIn1;
  end

  // ---------------------------------------------------------------------------
  // Add multiplier outputs
  assign mulAddOut = {mulAOut[2*wx-2], mulAOut} + {mulBOut[2*wx-2], mulBOut};

  // Update processing counter
  always @(posedge clk) begin
    procCxAdd <= procCxMul;
  end
  
  // Accumulator updates
  always @(*) begin

    // Summation, use sign extended multiplication output
    if (runFfe) begin
      if (cnt==1 | cnt==2) begin
        addIn0 = 0;
      end else 
      if (procCxAdd==1) begin
        addIn0 = accRe;
      end else begin
        addIn0 = accIm;
      end
      addIn1 = {{6{mulAddOut[2*wx-1]}}, mulAddOut};
    
    end else begin
      if (cnt==0 | runShFac | runChCor) begin
        addIn0 = 0;
      end else begin
        addIn0 = accRe;
      end
      addIn1 = {6'b000000, mulAddOut};
    end
    
    addOut = addIn0 + addIn1;
    
  end

  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin
    
    // Reset
    if (run1==0) begin
      yC_orR  <= 0;

    // Normal operation
    end else begin
      
      // Update of first adder stage
      if (x_reD & (procCxAdd==1 | runAmpEst | runShFac)) begin
        accRe <= addOut;
      end else 
      if (procCxAdd==2) begin
        accIm <= addOut;
      end
      
      // Assign output ready
      if (runChCor & procCxAdd==2) begin
        yC_orR <= 1;
      end else begin
        yC_orR <= 0;
      end

    end // if (run1==0) ... else
  end // always

  // --------------------------------------------------------------------------------------
  // Assign output registers
  assign resRdy = (runFfe    & cnt==0           & x_reDD) | 
                  (runAmpEst & cntP1==numSmpl_q & x_reD); 
                   
  assign yReLo_weint = resRdy;
  assign yImHi_weint = resRdy;
  
  assign yReLo_d = runFfe ? accRe[16+sh1-1:sh1]  : addOut[16+sh2-1:sh2];
  assign yImHi_d = runFfe ? addOut[16+sh1-1:sh1] : addOut[2*wx-1+6:16+sh2];

  // Assign output shFac
  assign ySh    = accRe[wy+sh3-1:sh3];
  assign ySh_or = runShFac & x_reDD;

  // Assign channel corrected output
  assign yC    = {accIm[wc+sh4-1:sh4], accRe[wc+sh4-1:sh4]};
  assign yC_or = yC_orR;

  // Assign output interrupt (ChCor IRQ removed, not needed currently)
  assign irqDone = resRdy | 
                   (runShFac & cntP1[6:0]==numSmpl_q & x_reD);// |
                   //(runChCor & cnt==0 & procCxAdd==0 & yC_or);

endmodule
//=========================================================================================

