//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   cordicR2P
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module cordicR2P(irqDone, 
                 regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters

  parameter irqDone_w         = 0;

  parameter run1_w            = 0;
  parameter run1_r            = 0;
  parameter run1_s            = 0;
  parameter R_run1            = 0;
  parameter numIt_w           = 0;
  parameter numIt_r           = 0;
  parameter numIt_s           = 0;
  parameter R_numIt           = 0;
  parameter xReAbs_w          = 0;
  parameter xReAbs_r          = 0;
  parameter xReAbs_s          = 0;
  parameter R_xReAbs          = 0;
  parameter xImPh_w           = 0;
  parameter xImPh_r           = 0;
  parameter xImPh_s           = 0;
  parameter R_xImPh           = 0;
  parameter thr_w             = 0;
  parameter thr_r             = 0;
  parameter thr_s             = 0;
  parameter R_thr             = 0;

  parameter wx                = 0;
  parameter w                 = 0;
  parameter numItMax          = 0;
  parameter ph0               = 0;
  parameter ph1               = 0;
  parameter ph2               = 0;
  parameter ph3               = 0;
  parameter ph4               = 0;
  parameter ph5               = 0;
  parameter ph6               = 0;
  parameter ph7               = 0;
  parameter ph8               = 0;
  parameter ph9               = 0;
  parameter ph10              = 0;
  parameter ph11              = 0;
  parameter ph12              = 0;
  parameter ph13              = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  output      [irqDone_w-1:0]  irqDone;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire           [run1_w-1:0]  run1_q;
  wire           [run1_w-1:0]  run1_d;
  wire                         run1_weint;
  assign                       run1_weint = 0;

  wire          [numIt_w-1:0]  numIt_q;
  wire          [numIt_w-1:0]  numIt_d;
  wire                         numIt_weint;
  assign                       numIt_weint = 0;

  wire         [xReAbs_w-1:0]  xReAbs_q;
  wire         [xReAbs_w-1:0]  xReAbs_d;
  wire                         xReAbs_weint;

  wire          [xImPh_w-1:0]  xImPh_q;
  wire          [xImPh_w-1:0]  xImPh_d;
  wire                         xImPh_weint;

  wire            [thr_w-1:0]  thr_q;
  wire            [thr_w-1:0]  thr_d;
  wire                         thr_weint;
  assign                       thr_weint = 0;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg run1(run1_q, regBus, run1_d, run1_weint, regWe, regRe, clk, reset);
  defparam run1.w    = run1_w;
  defparam run1.rval = run1_r;
  defparam run1.sgn  = run1_s;
  defparam run1.adr  = R_run1;

  ereg numIt(numIt_q, regBus, numIt_d, numIt_weint, regWe, regRe, clk, reset);
  defparam numIt.w    = numIt_w;
  defparam numIt.rval = numIt_r;
  defparam numIt.sgn  = numIt_s;
  defparam numIt.adr  = R_numIt;

  ereg xReAbs(xReAbs_q, regBus, xReAbs_d, xReAbs_weint, regWe, regRe, clk, reset);
  defparam xReAbs.w    = xReAbs_w;
  defparam xReAbs.rval = xReAbs_r;
  defparam xReAbs.sgn  = xReAbs_s;
  defparam xReAbs.adr  = R_xReAbs;

  ereg xImPh(xImPh_q, regBus, xImPh_d, xImPh_weint, regWe, regRe, clk, reset);
  defparam xImPh.w    = xImPh_w;
  defparam xImPh.rval = xImPh_r;
  defparam xImPh.sgn  = xImPh_s;
  defparam xImPh.adr  = R_xImPh;

  ereg thr(thr_q, regBus, thr_d, thr_weint, regWe, regRe, clk, reset);
  defparam thr.w    = thr_w;
  defparam thr.rval = thr_r;
  defparam thr.sgn  = thr_s;
  defparam thr.adr  = R_thr;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0;
  assign regReOut = 0;

// ----------------------------------------------------------------------------------------
//=========================================================================================

  // --------------------------------------------------------------------------------------
  // Register definitions
  reg           [1:0]  cntIn;
  reg           [1:0]  cnt;
  reg           [3:0]  cntIt;
  reg signed  [w-1:0]  xRe, xReBuf, xIm, phi, xA, xB, xS0, xS, p, xP, xM, 
                       addIn0, addIn1, addOut; 
  reg                  xReWe, xImWe;

  // --------------------------------------------------------------------------------------
  // Assign input and output enable
  assign irqDone      = cntIt==numIt_q+2;
  assign xReAbs_weint = cntIt==numIt_q+2;
  assign xImPh_weint  = cntIt==numIt_q+2;
  assign xReAbs_d     = xReBuf;
  assign xImPh_d      = {phi, {16-w{1'b0}}};
  
  // --------------------------------------------------------------------------------------
  // Signal processing behaviour
  always @(*) begin
        
    // xRe/xIm exchange mux
    if (cnt==0) begin
      xA = xRe;
      xB = xIm;
    end else begin
      xA = xIm;
      xB = xRe;
    end
    
    // Shifter
    case (cntIt)
      0: xS0 = xB;
      1: xS0 = xB;
      2: xS0 = xB>>>1;
      3: xS0 = xB>>>2;
      4: xS0 = xB>>>3;
      5: xS0 = xB>>>4;
      6: xS0 = xB>>>5;
      7: xS0 = xB>>>6;
      8: xS0 = xB>>>7;
      9: xS0 = xB>>>8;
     10: xS0 = xB>>>9;
     11: xS0 = xB>>>10;
     12: xS0 = xB>>>11;
     13: xS0 = xB>>>12;
     14: xS0 = xB>>>13;
     default: xS0 = xB; 
    endcase
    
    // Apply rounding towards 0 after shifting
    if (cntIt==0 | cntIt==1) begin
      xS = xB;
    end else begin
      xS = xS0[w-1] ? xS0+1 : xS0;
    end
        
    // Phase LUT
    case (cntIt)
      0: p = ph0;
      1: p = ph1;
      2: p = ph2;
      3: p = ph3;
      4: p = ph4;
      5: p = ph5;
      6: p = ph6;
      7: p = ph7;
      8: p = ph8;
      9: p = ph9;
     10: p = ph10;
     11: p = ph11;
     12: p = ph12;
     13: p = ph13;
     default: p = ph0; 
    endcase
    
    if (cntIt==numIt_q+1) begin
      xP = {2'b00, thr_q};
    end else
    if (cnt==1) begin
      xP = p;
    end else begin
      xP = xS;
    end
    
    xM = -xP;
    
    // Adder input 0
    if (cntIt==0) begin
      addIn0 = 0;
    end else
    if (cnt==1) begin
      addIn0 = phi;
    end else begin
      addIn0 = xA;
    end

    // Adder input 1
    if ((cnt==2 ^ xIm[w-1]) | cntIt==numIt_q+1) begin
      addIn1 = xM;
    end else begin
      addIn1 = xP;
    end
    
    // Adder
    addOut = addIn0 + addIn1;
    
  end

  // --------------------------------------------------------------------------------------
  // FSM
  always @(posedge clk) begin

    // Assign write enables
    xReWe <= regBus==R_xReAbs & regWe;
    xImWe <= regBus==R_xImPh  & regWe;
    
    // Reset all state when writing to xRe
    if (run1_q==0 | xReWe) begin
      cnt   <= 0;
      cntIt <= 0;
      cntIn <= 0;
      
    // Normal operation
    end else begin
      
      // Counter updates
      if (xImWe) begin              // The input counter counts up while data is
        cntIn <= 2;                 // being read from the input, and is reset at
      end else                      // the end of the Cordic operation
      if (cntIt==numIt_q+2) begin
        cntIn <= 0;
      end
      
      if (cnt==2 | cntIt==numIt_q+2) begin
        cnt <= 0;
      end else                      // The inner iteration counter counts 0,1,2,0,...
      if (cntIn==2) begin           // when data is in the input, and continues until
        cnt <= cnt+1;               // the end of Cordic operation
      end
      
      if (cntIt==numIt_q+2) begin
        cntIt <= 0;
      end else
      if (cnt==2 | cntIt==numIt_q+1) begin
        cntIt <= cntIt+1;
      end
      
      // Register update using input - load xRe/xIm serially
      if (xImWe) begin
        xRe  <= xReAbs_q;
        xIm  <= regBus[w-1:0];
      end else
      
      // Update registers - depends on internal states
      if (cnt==0) begin
        xReBuf <= addOut;
      end else 
      if (cnt==1) begin
        phi <= addOut;
      end else begin
        xIm <= addOut;
        xRe <= xReBuf;
      end

    end // if (run1_q==0) ... else
  end // always

endmodule
//=========================================================================================

