function [cfm, rxParam] = msgWlanRxRun(id, timerMode, rxTime)
%===============================================================================
%  Copyright (C) BAY9, 2011
%===============================================================================
%
% Syntax:
%   [cfm rxParam] = msgWlanRxRun(id, timerMode, rxTime)
%
% Inputs:
%   id          - Target identifier
%   timerMode   - 0: run until stopped externally
%                 1: run for the given number of sample ticks
%                 2: run until the given time is reached
%   rxTime      - 0..2^32-1
%
% Outputs:
%   cfm         - Combined RX confirm messages
%   rxParams    - RX parameters
%    .hdrErr     - 0/1 header OK / error
%    .mode       - 0-7 = 6..54 MBit/s
%    .len        - RX length in bytes
%    .fOffHz     - Frequency offset in Hz
%    .amp        - Estimated BB amplitude
%    .agcAttn    - AGC attenuation setting
%    .startTime  - Start time (32 bit)
%    .crc        - CRC result (only if hdrErr=0)
%
% Description:
%   Send RX request message and wait for confirms using subfunctions
%   msgWlanRxRun1 and msgWlanRxRun2. While msgWlanRxRun1 initiates
%   reception and reads the RX start confirm message, msgWlanRxRun2 is 
%   handles the rest of the RX FSM after  the RX start confirm message
%   was received. See subfunctions for details.
%
%   The variable rxParam is set according the received confirm messages.
%
%===============================================================================

  % ----------------------------------------------------------------------------
  % First, send start request and handle immediate confirm
  cfmRxStart = msgWlanRxRun1(id, timerMode, rxTime);  % Start message only
  
  % Second, handle the rest of the RX FSM
  [cfmRxAcqEnd cfmRxHdr cfmRxEnd] = msgWlanRxRun2(id);
  
  % Assemble confirm output
  cfm = [cfmRxStart cfmRxAcqEnd cfmRxHdr cfmRxEnd];
  
  % ----------------------------------------------------------------------------
  % Result evaluation
  rxParam.acqStop = (cfmRxAcqEnd(2) ~= 2);
  if (rxParam.acqStop)
    rxParam.acqStopMode = cfmRxAcqEnd(2);
  else
    rxParam.hdrErr      = cfmRxHdr(2);
    rxParam.mode        = cfmRxHdr(3);
    rxParam.len         = cfmRxHdr(4);
    fOff                = cfmRxHdr(5) - 2^16*(cfmRxHdr(5)>=2^15);
    rxParam.fOffHz      = round(fOff / 2^13 * 312500);
    rxParam.amp         = cfmRxHdr(6);
    rxParam.agcAttn     = cfmRxHdr(7);
    rxParam.startTime   = cfmRxHdr(8) +  2^16*cfmRxHdr(9);
    rxParam.crc         = cfmRxEnd(2);
  end

% ------------------------------------------------------------------------------

