function stat = testTxRx11g(numFrames, modeTx, lenTx, c2nDb, printResults)
%===============================================================================
%  Copyright (C) BAY9, 2013
%===============================================================================
%
% Syntax:
%   testTxRx11g(numFrames, modeTx, lenTx, c2nDb, printResults)
%
% Inputs:
%   numFrames     - Number of frames to test
%   modeTx        - TX mode, 0..7
%   lenTx         - TX length in bytes, 1..4095
%   c2nDb         - Carrier-to-noise ratio in dB, 0..100
%   printResults  - 0/1 -> do not / do print results statistics
%
% Outputs:
%   stat          - Error statistic, struct with fields
%                    .crcOk
%                    .crcErr
%                    .hdrErr
%                    .noAcq
%
% Description:
%   Test function to transmit an receive frames using the VRF. Except for
%   numFrames, all inputs can be vectors. The setting for each transmission
%   are then choses randomly from the vector entries.
% 
%   The function executes the following steps:
%     - Set the channel gain to achieve the desired carrier signal to noise ratio
%     - Start RX for WLAN instance 1
%     - Start TX for WLAN instance 0 (use internal data generator)
%     - Read RX confirm messages, evaluate RX reception
%     - Display RX results for each single frame:
%       o C2N:    Carrier to noise ratio
%       o M:      RX mode (0: BPSK R=1/2, 1: BPSK R=3/4, ..., 7: 64-QAM R=3/4)
%       o Len:    Length in bytes
%       o Amp:    Amplitude input level
%       o AGC:    AGC step taken (roughly attenuation in dB)
%       o fOffHz: Measured frequency offset in Hz
%       o C:      CRC status 
%                   0 = CRC OK
%                   1 = CR Error
%                   2 = Header error
%                   3 = Acquisition failed, no RX frame
%

  % ----------------------------------------------------------------------------
  % Check input arguments, possible set some defaults
  if (nargin < 1 || isempty(numFrames))
    numFrames = 100;
  end
  if (nargin < 2 || isempty(modeTx))
    modeTx = [0:7];
  end
  if (nargin < 3 || isempty(lenTx))
    lenTx = [1:4095];
  end
  if (nargin < 4 || isempty(c2nDb))
    c2nDb = [2:0.1:40];
  end
  if (nargin < 5 || isempty(printResults))
    printResults = 0;
  end
  
  % ----------------------------------------------------------------------------
  % Global definitions
  global MsgIdWlan;
  global idStd;
  if (isempty(idStd))
    idStd.tgt   = 'fpga0';
    idStd.top   = 'vrfX';
    idStd.inst  = 0;
  end
  
  idVrf     = idStd;
  idWlanTx  = idStd;
  idWlanRx  = idStd;
  idWlanRx.inst = 1;    % Overwrite instance
  
  % ----------------------------------------------------------------------------
  % TX timing
  txReqDelay      = 30;   % In samples
  txTimerMode     = 1;

  % RX timing
  rxTimeSec   = 0.5;
  rxTime32    = round(rxTimeSec*20e6);
  rxTimerMode = 1;

  % Keep some statistics
  crcAll = zeros(1, numFrames);

  % Set normal data source, just in case it has been changed
  msgWlanCfgTxDataSrcReq(idWlanTx, 0, 0, 0);

  % ----------------------------------------------------------------------------
  % Main burst loop
  for k = 0:numFrames-1

    % --------------------------------------------
    % Setup channel
    c2nDbC    = c2nDb(floor(length(c2nDb)*rand)+1); % Carrier to noise
    snrDbTd   = c2nDbC - 10*log10(256/52);          % Time domain SNR @ 80 MHz bandwidth
    cfm       = msgVrfCfgChGainReq(idVrf, snrDbTd); % Set SNR
    
    % --------------------------------------------
    % Prepare RX buffer - allow write from input, just use maximum size
    cfm = msgVrfDatBufRxStartReq(idVrf, 2047);
    
    % --------------------------------------------
    % Get current TX settings, start TX
    modeTxC = modeTx(floor(length(modeTx)*rand)+1);
    lenTxC  = lenTx(floor(length(lenTx)*rand)+1);
    
    dataSrc = 'externalGen';
%    dataSrc = 'externalFill';
%    dataSrc = 'internal';
    switch (dataSrc)

      % Use VRF external data generator to fill the buffer
      case 'externalGen'
        initDat = k+10;
        cntRnd  = 0;round(rand);
        endZero = 1;round(rand);
        msgVrfDatBufTxFillReq(idVrf, lenTxC, initDat, cntRnd, endZero);
        msgVrfDatBufTxStartReq(idVrf, ceil(lenTxC/2));

      % Use internal WLAN data source - disabled
      case 'internal'
        initDat = k+10;
        cntRnd  = 0;    % Only needed for check
        endZero = 1;    % Only needed for check
        msgWlanCfgTxDataSrcReq(idWlanTx, 1, lenTxC, initDat);

      % Write manually to buffer - disabled
      case 'externalFill'
        initDat = k+10;
        cntRnd  = 0;    % Only needed for check
        endZero = 1;    % Only needed for check
        dat = [mod(initDat+[0:lenTxC-5]*1, 256) 0 0 0 0];
        dat = [dat hex2dec('BB')*ones(1, mod(lenTxC, 2))];
        dat = dat(1:2:end) + 2^8*dat(2:2:end);
       
        msgVrfDatBufTxWriteReq(idVrf, dat);            
        msgVrfDatBufTxStartReq(idVrf, ceil(lenTxC/2));         
    
    end

    % --------------------------------------------    
    % Send initial RX start request + read start cfm
    cfmRxStart  = msgWlanRxRun1(idWlanRx, rxTimerMode, rxTime32);
    
    % --------------------------------------------    
    % Start TX
    msgWlanTxRun(idWlanTx, modeTxC, lenTxC, txTimerMode, txReqDelay);

    % --------------------------------------------
    % Evaluate remaining RX confirm messages
    [cfmRxAcqEnd, cfmRxHdr, cfmRxEnd] = msgWlanRxRun2(idWlanRx);

    % --------------------------------------------------------------------------
    % Evaluate RX return parameters
    [rxp, rxpS] = ...
      testRxReqEval(cfmRxAcqEnd, cfmRxHdr, cfmRxEnd);
    
    % --------------------------------------------------------------------------
    % Check RX data buffer, overwrite original rxp.crc
    %  -> works also for last 4 bytes ~= 0 
    externalSnk = 1;
    if (externalSnk & rxp.crc < 2)
      rxp.crc   = msgVrfDatBufRxCheckReq(idVrf, lenTxC, initDat, cntRnd, endZero);
      rxpS.crc  = sprintf('%d', rxp.crc);
    end

    % CRC + Carrier to noise ratio
    c2nS = sprintf('%4.1f', c2nDbC);
    
    % Keep CRC errors
    crcAll(k+1) = rxp.crc;
    
    % --------------------------------------------------------------------------
    % Print header each 20 frames
    if (mod(k, 20)==0)
      fprintf('\n');
      fprintf('+--------+------+---+------+-----+-----+---------+---+\n');
      fprintf('|   No   |  C2N | M |  Len | Amp | AGC |  fOffHz | C |\n');
      fprintf('+--------+------+---+------+-----+-----+---------+---+\n');
    end
    fprintf('| %6d | %s | %s | %s | %s | %s | %s | %s |\n', ...
              k, c2nS, rxpS.rxMode, rxpS.rxLen, rxpS.amp, rxpS.agcAttn, ...
                 rxpS.fOffHz, rxpS.crc);

    % --------------------------------------------------------------------------
    % Read/print RX data buffer - normally disabled, only for debugging
    if 0
      dat = msgVrfDatBufRxReadReq(idVrf, ceil(lenTxC/2));
      dat = [mod(dat, 2^8); floor(dat/2^8)];
      dat = dat(:)';
      dat = dat(1:lenTxC)
    end
    
  end
  
  % --------------------------------------------------------------------------
  % Calculate statistics
  stat.crcOk  = sum(crcAll==0) / numFrames;
  stat.crcErr = sum(crcAll==1) / numFrames;
  stat.hdrErr = sum(crcAll==2) / numFrames;
  stat.noAcq  = sum(crcAll==3) / numFrames;

  if (printResults)
    fprintf('------------------------------------------------------\n');
    fprintf('\n');
    fprintf('  CRC OK:  %5.1f%%\n', stat.crcOk *100);
    fprintf('  CRC Err: %5.1f%%\n', stat.crcErr*100);
    fprintf('  Hdr Err: %5.1f%%\n', stat.hdrErr*100);
    fprintf('  No  Acq: %5.1f%%\n', stat.noAcq *100);
    fprintf('\n');
  end

% ------------------------------------------------------------------------------
