function setupVrfPa(id, bypass, backOffDb, phOffMaxDeg)
%===============================================================================
%  Copyright (C) BAY9, 2014
%===============================================================================
%
% Syntax:
%   setupVrfPa(id, bypass, backOffDb, phOffMaxDeg)
%
% Input:
%   id          - Target ID
%   bypass      - 0/1 = bypass LUT off/on
%   backOffDb   - [0..20], PA power backoff in dB
%   phOffMaxDeg - [-180..180], maximum phase shift in degree
%
% Output:
%   --
%
% Description:
%   The function provides an example of how to configure the PA LUTs. It
%   first configures the bypass of the PA model. In case the bypass is
%   set to 0, the LUT for amplitude factor and phase offset are calculated
%   and send via message CfgTxPaLutReq to the VRF.
%
%   For amplitude modeling, a tangens hyperbolicus function is used to 
%   define the dependency between input amplitude x and output amplitude y 
%   of the PA:
%
%     y = ampMax*tanh(x/ampMax)
%
%   where ampMax = rmsIn * 10^(backOffDb/20) is set to the maximum output 
%   level of the PA module. Similar to a real PA, the function is close to 
%   y = x for small x, while converging to ampMax for large x. Value rmsIn
%   is a constant having the value of to the expected input level of the PA 
%   model which is -20 dB full scale of the 16 bit range, i.e 
%   rmsIn = 32768 * 10^(-20/20) = 3276.8
%   
%   For phase offset modeling, the following function, linear in terms of
%   decibel input scale is used:
%
%                 { 0                         for xDb <  -20
%     phOffDeg =  { phOffMaxDeg*(xDb+20)/40   for xDb >= -20
%
%   This provides zero phase shift for input levels x < -20 dB, and a phase shift
%   that is linear with the input level in dB for input levels from -20..+20 dB.
%
%   Both functions are chosen arbitrarily for demonstration purposes only and do 
%   not necessarily model a PA correctly.
%

  % ----------------------------------------------------------------------------
  % Check input arguments
  if (nargin < 1)
    id = 0;
  end
  if (nargin < 2)
    bypass = 0;
  end
  if (nargin < 3)
    backOffDb = 9;
  end
  if (nargin < 4)
    phOffMaxDeg = 10;
  end
  
  % ----------------------------------------------------------------------------
  % Define message
  global MsgIdVrf;
  
  % Switch on/off debug plots
  dbgPlot = 0;
  
  % ----------------------------------------------------------------------------
  % Setup bypass
  msgVrfCfgTxPaReq(id, bypass);

  % ----------------------------------------------------------------------------
  % Setup LUT only if bypass not used
  if (bypass == 0)

    % Calculate amplitude factor LUT entries based on function
    %
    %   y = a*tanh(x/a)
    %
    rmsIn    = 3276.8;                      % Default PA model input RMS level
    ampMax   = rmsIn * 10^(backOffDb/20);   % Max output level, backOffDb vs average
    x        = [0:64:32768];                % Input level LUT values
    x(1)     = 1e-10;                       % Avoid division/log problems
    y        = ampMax*tanh(x/ampMax);       % Output level after PA modeling
    ampFac   = y./x;                        % Corresponding scaling factors
    ampFac   = min(ampFac, 1);              % Fix rounding errors with ampFac > 1

    % Calculate phase shift LUT entries bases on function
    %
    %     { 0                         for xDb <  -20
    % y = { phOffMaxDeg*(xDb+20)/40   for xDb >= -20
    %
    xDb               = 20*log10(x/rmsIn);            % Get inputs x in dB
    phOffDeg          = phOffMaxDeg * (xDb+20) / 40;  % Get corresponding
    phOffDeg(xDb<-20) = 0;                            % phase offsets

    % Send message to set LUT finally
    msgVrfCfgTxPaLutReq(id, ampFac, phOffDeg);

    % --------------------------------------------------------------------------
    % Debug plot, output vs. input power, 0 dB = average power level
    if (dbgPlot)
      figure(1);
      subplot(1,2,1);
      xDb = 20*log10(x/rmsIn);
      yDb = 20*log10(y/rmsIn);
      hold off;
      plot(xDb, yDb, 'b');
      hold on;
      plot(xDb, backOffDb*ones(size(xDb)), 'r--');
      grid on
      xlim([-35, 20]);
      ylim([-35, 20]);
      xlabel('Input power [dB]');
      ylabel('Output power [dB]');
      legend('Transfer curve', 'PA limit', 'Location', 'SouthEast');
      title('Amplitude compression');
    end

    % Debug plot, output phase offset vs. input power, 0 dB = average power level
    if (dbgPlot)
      figure(1);
      subplot(1,2,2);
      hold off;
      plot(xDb, phOffDeg, 'b');
      grid on
      xlim([-35, 20]);
      ylim([-100, 100]);
      xlabel('Input power [dB]');
      ylabel('Output phase offset [deg]');
      legend('Transfer curve', 'Location', 'SouthEast');
      title('Phase deviation');
    end

  end
  
% ------------------------------------------------------------------------------
