function cfm = sendMsg(id, req, nCfm)
%===============================================================================
%  Copyright (C) BAY9, 2014
%===============================================================================
%
% Syntax:
%   cfm = sendMsg(id, req, nCfm)
%
% Inputs:
%   id      - Target identifier
%    .core    - 'wlan' / 'vrf'  -> Current module within the core
%    .inst    - Instance number of a certain core (if there are several)
%   req     - Request message
%   nCfm    - Number of expected confirms
%
% Outputs:
%   cfm     - Confirm message
%
% Description:
%   Send message to VRF or WLAN subsystem. In case of VRF simply send
%   message directly via the UART device file. For WLAN, use the
%   message forwarding mechanism of the VRF, i.e. VRF messages CtrlWr/Rd.
%
%===============================================================================

  % ----------------------------------------------------------------------------
  % For VRF target -> send message directly
  if (strcmp(id.core, 'vrf'))

    % ------------------------------------
    % Auto init UART connection if not yet established
    persistent uartFid;
    if (isempty(uartFid))
      uartFid = uartInit();
    end

    % ------------------------------------
    % Send request
    if (isempty(req)==0)
      fwrite(uartFid, mod(req, 2^16), 'uint16');
      fflush(uartFid);
    end

    % Read confirm message
    if (nCfm)
      cfm = fread(uartFid, nCfm, 'uint16')';
    else
      cfm = [];
    end

  % ----------------------------------------------------------------------------
  % For WLAN target -> pass to WLAN cores using VRF Ctrl messages
  elseif (strcmp(id.core, 'wlan'))

    % Redirect
    ctrlIdx  = id.inst;   % Instance ID -> control interface index
    id.inst  = 0;         % VRF instance 0

    % Send request
    if (length(req) > 0)
      msgVrfCtrlWrReq(id, ctrlIdx, req);
    end

    % Get confirm
    if (nCfm > 0)
      cfm = msgVrfCtrlRdReq(id, ctrlIdx, nCfm);
    else
      cfm = [];
    end
  end

%===============================================================================

