//====================================================================================
//  Copyright (C) BAY9, 2014
//====================================================================================
//
// MODULE:
//   vrf
//
// PURPOSE:
//   Top module
//
// INPUT/OUTPUT:
//   Control input/output: 
//     uartOut:             out: UART transmit signal
//     uartIn:              in:  UART receive signal
//     ctrlOut:             out: External control interface (16 bit), output
//     ctrlOut_or:          out: Output ready signal
//     ctrlOut_ir:          in:  Input ready signal
//     ctrlIn:              in:  External control interface (16 bit), input
//     ctrlIn_or:           in:  Output ready signal
//     ctrlIn_ir:           out: Input ready signal
//     ctrlSel:             in:  Control select, 0=CtrlIf, 1=UART
//   Misc:    
//     gpo:                 out: General purpose output (16 bit)
//     gpi:                 in:  General purpose input (16 bit)
//     statLed:             out: Status LEDs (8 bit)
//   ADC/DAC:
//     adcRe/Im:            out: ADC outputs, real/imag part, 12 bit signed to PHY1-RX
//     adc_or:              out: ADC handshake (input), adc_or=1 indicates data is written
//     dacRe/Im:            in:  DAC inputs, real/imag part, 12 bit signed from PHY0-TX
//     dac_ir:              out: DAC handshake (input), dac_ir=1 indicates data is read
//   Gain selction:
//     gainSel:             in:  Gain table selection from PHY1-RX
//   PHY control - messages:
//     ctrlPhy0/1Out:       out: Write 16 bit message to PHY0/1
//     ctrlPhy0/1Out_or:    out: Output ready signal to PHY0/1
//     ctrlPhy0/1Out_ir:    in:  Input ready signal from PHY0/1
//     ctrlPhy0/1In:        in:  Read 16 bit message to PHY0/1
//     ctrlPhy0/1In_or:     in:  Output ready signal from PHY0/1
//     ctrlPhy0/1In_ir:     out: Input ready signal to PHY0/1
//   PHY control - data:
//     dataPhy0Tx:           out: Write data to PHY0-TX
//     dataPhy0Tx_ir:        in:  Input ready signal from PHY0-TX
//     dataPhy1Rx:           in:  Read data from PHY1-RX
//     dataPhy1Rx_or:        in:  Output ready signal from PHY1-RX
//   Clock/reset:
//     clk:                 in:  Clock
//     resetIn:             in:  Reset (active high)
//     resetOut:            out: Indicates that system is still in reset
//
// DESCRIPTION:
//   Module vrf.v provide a wrapper for the autogenerated module main.v with
//   the following functionality:
//    - Order inputs/output according to functionality
//    - Include auto-reset circuit (resetCnt)
//    - Provide register bus control feedback
//
// HISTORY:
//   08-Oct-14, Dirk Sommer
//     Initial version
//
//====================================================================================

module vrf(uartOut, 
           uartIn, 
           ctrlOut, ctrlOut_or, ctrlOut_ir, 
           ctrlIn,  ctrlIn_or,  ctrlIn_ir, 
           ctrlSel, 
           
           gpo,
           gpi,
           statLed,
           
           adcRe,
           adcIm, 
           adc_or,
           
           dacRe,
           dacIm,
           dac_ir,

           gainSel,

           ctrlPhy0Out, ctrlPhy0Out_or, ctrlPhy0Out_ir, 
           ctrlPhy0In,  ctrlPhy0In_or,  ctrlPhy0In_ir, 
           ctrlPhy1Out, ctrlPhy1Out_or, ctrlPhy1Out_ir, 
           ctrlPhy1In,  ctrlPhy1In_or,  ctrlPhy1In_ir, 

           dataPhy0Tx, dataPhy0Tx_or, dataPhy0Tx_ir,
           dataPhy1Rx, dataPhy1Rx_or, dataPhy1Rx_ir,

           resetIn, resetOut, clk);

  // --------------------------------------------------------------------------------------
  // Include global definitions
  `include "def_Const_vrf.v"

  // --------------------------------------------------------------------------------------
  // Inputs and outputs

  // -----------------------------------
  // External interface + UART control, debug LEDs
  output                          uartOut;
  input                           uartIn;

  output   [VrfCtrlWidth_C-1:0]   ctrlOut;
  output                          ctrlOut_or;
  input                           ctrlOut_ir;

  input    [VrfCtrlWidth_C-1:0]   ctrlIn;
  input                           ctrlIn_or;
  output                          ctrlIn_ir;

  input                           ctrlSel;
  
  output    [VrfGpioWidth_C-1:0]  gpo;
  input     [VrfGpioWidth_C-1:0]  gpi;
  output [VrfStatLedWidth_C-1:0]  statLed;

  // -----------------------------------
  // Connections to PHY under test

  // ADC output
  output  [VrfAdcDacWidth_C-1:0]  adcRe;
  output  [VrfAdcDacWidth_C-1:0]  adcIm;
  output                          adc_or;
  
  // DAC input
  input   [VrfAdcDacWidth_C-1:0]  dacRe;
  input   [VrfAdcDacWidth_C-1:0]  dacIm;
  output                          dac_ir;

  // Gain selection, PA on, etc..
  input  [VrfGainSelWidth_C-1:0]  gainSel;

  // Control of PHY0
  output   [VrfCtrlWidth_C-1:0]   ctrlPhy0Out;
  output                          ctrlPhy0Out_or;
  input                           ctrlPhy0Out_ir;

  input    [VrfCtrlWidth_C-1:0]   ctrlPhy0In;
  input                           ctrlPhy0In_or;
  output                          ctrlPhy0In_ir;

  // Control of PHY1
  output   [VrfCtrlWidth_C-1:0]   ctrlPhy1Out;       
  output                          ctrlPhy1Out_or;    
  input                           ctrlPhy1Out_ir;    

  input    [VrfCtrlWidth_C-1:0]   ctrlPhy1In;
  input                           ctrlPhy1In_or;
  output                          ctrlPhy1In_ir;

  // Data of PHY0
  output    [VrfDataWidth_C-1:0]  dataPhy0Tx;
  output                          dataPhy0Tx_or;
  input                           dataPhy0Tx_ir;
  
  // Data of PHY1
  input     [VrfDataWidth_C-1:0]  dataPhy1Rx;
  input                           dataPhy1Rx_or;
  output                          dataPhy1Rx_ir;

  // -----------------------------------
  // Define clock and reset
  input                        clk;         // Clock
  input                        resetIn;     // Reset input
  output                       resetOut;    // Reset output

  // ----------------------------------------
  // Unused clock signals
  wire          clk2;

  // RegBus signals
  wire   [15:0] regBus;
  wire          regWe;
  wire          regRe;

  // Internal reset lines
  wire          reset;

  // ===========================================================================
  // Define main module
  `include "rfMain_0.v"

  // ---------------------------------------------------------------------------
  // Set unused clock signals to 0
  assign clk2                         = 0;
  
  // ---------------------------------------------------------------------------
  // Register bus controls feedback
  assign regWe                        = rfMain_0_regWe;
  assign regRe                        = rfMain_0_regRe;

  // ---------------------------------------------------------------------------
  // Connect UART, control interface, and control selector
  assign  uartOut                     = rfMain_0_uartOut;
  assign  rfMain_0_uartIn             = uartIn;

  assign  ctrlOut                     = rfMain_0_ctrlOut;
  assign  ctrlOut_or                  = rfMain_0_ctrlOut_or;
  assign  rfMain_0_ctrlOut_ff         = ~ctrlOut_ir;
  
  assign  rfMain_0_ctrlIn             = ctrlIn;
  assign  rfMain_0_ctrlIn_fe          = ~ctrlIn_or;
  assign  ctrlIn_ir                   = rfMain_0_ctrlIn_ir;
  
  assign  rfMain_0_ctrlSel            = ctrlSel;

  // ---------------------------------------------------------------------------
  // Connect status LEDs, GPO, and GPI
  assign  gpo                         = rfMain_0_gpo;
  assign  rfMain_0_gpi                = gpi;
  assign  statLed                     = rfMain_0_statLed;

  // ---------------------------------------------------------------------------
  // Connect ADC/DAC
  assign  adcRe                       = rfMain_0_adcRe;  
  assign  adcIm                       = rfMain_0_adcIm;
  assign  adc_or                      = rfMain_0_adcRe_or & rfMain_0_adcIm_or;
  
  assign  rfMain_0_dacRe              = dacRe;
  assign  rfMain_0_dacIm              = dacIm;
  assign  dac_ir                      = rfMain_0_dacRe_ir & rfMain_0_dacIm_ir;
  
  // Connect control
  assign  ctrlPhy0Out                 = rfMain_0_ctrlPhy0Out;    
  assign  ctrlPhy0Out_or              = rfMain_0_ctrlPhy0Out_or; 
  assign  rfMain_0_ctrlPhy0Out_ff     = ~ctrlPhy0Out_ir;

  assign  rfMain_0_ctrlPhy0In         = ctrlPhy0In;             
  assign  rfMain_0_ctrlPhy0In_fe      = ~ctrlPhy0In_or;
  assign  ctrlPhy0In_ir               = rfMain_0_ctrlPhy0In_ir;

  assign  ctrlPhy1Out                 = rfMain_0_ctrlPhy1Out;    
  assign  ctrlPhy1Out_or              = rfMain_0_ctrlPhy1Out_or; 
  assign  rfMain_0_ctrlPhy1Out_ff     = ~ctrlPhy1Out_ir;
  
  assign  rfMain_0_ctrlPhy1In         = ctrlPhy1In;             
  assign  rfMain_0_ctrlPhy1In_fe      = ~ctrlPhy1In_or;
  assign  ctrlPhy1In_ir               = rfMain_0_ctrlPhy1In_ir;

  // Connect data
  assign  dataPhy0Tx                  = rfMain_0_dataPhy0Tx;    
  assign  dataPhy0Tx_or               = rfMain_0_dataPhy0Tx_or;
  assign  rfMain_0_dataPhy0Tx_ff      = ~dataPhy0Tx_ir;
 
  assign  rfMain_0_dataPhy1Rx         = dataPhy1Rx;             
  assign  rfMain_0_dataPhy1Rx_fe      = ~dataPhy1Rx_or;
  assign  dataPhy1Rx_ir               = rfMain_0_dataPhy1Rx_ir;

  // Connect misc
  assign  rfMain_0_gainSel            = gainSel;
  
  // ---------------------------------------------------------------------------
  // Reset circuit for power on and reset combine
  resetCnt resetCnt(reset, rfMain_0_resetOut | resetIn, clk);
  assign resetOut = reset;

endmodule

//==============================================================================

