//====================================================================================
//  Copyright (C) BAY9, 2016
//====================================================================================
//
// MODULE:
//   X16
//
// ----------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------

module X16(irqExt, 
           regBus, regWe, regRe, regWeOut, regReOut, clk1, clk2, reset);

  // --------------------------------------------------------------------------------------
  // Module parameters

  parameter irqExt_w          = 0;

  parameter cmd_w             = 0;
  parameter cmd_r             = 0;
  parameter cmd_s             = 0;
  parameter R_cmd             = 0;
  parameter pramAdr_w         = 0;
  parameter pramAdr_r         = 0;
  parameter pramAdr_s         = 0;
  parameter R_pramAdr         = 0;
  parameter pramDat_w         = 0;
  parameter pramDat_r         = 0;
  parameter pramDat_s         = 0;
  parameter R_pramDat         = 0;
  parameter dramAdr_w         = 0;
  parameter dramAdr_r         = 0;
  parameter dramAdr_s         = 0;
  parameter R_dramAdr         = 0;
  parameter dramDat_w         = 0;
  parameter dramDat_r         = 0;
  parameter dramDat_s         = 0;
  parameter R_dramDat         = 0;

  parameter RegStart          = 0;
  parameter RegLen            = 0;
  parameter RegEnd            = 0;
  parameter DramStart         = 0;
  parameter DramLen           = 0;
  parameter DramEnd           = 0;
  parameter DramAdrWidth      = 0;
  parameter PramStart         = 0;
  parameter PramLen           = 0;
  parameter PramLen32         = 0;
  parameter PramEnd           = 0;
  parameter PramAdrWidth      = 0;

  // --------------------------------------------------------------------------------------
  // Inputs and outputs
  input        [irqExt_w-1:0]  irqExt;

  // Define clock and reset
  input                        clk1;
  input                        clk2;
  input                        reset;

  // Inputs and outputs for registers
  inout                [15:0]  regBus;
  input                        regWe;
  input                        regRe;
  output                       regWeOut;
  output                       regReOut;

  // Wires for external registers
  wire            [cmd_w-1:0]  cmd_q;
  wire            [cmd_w-1:0]  cmd_d;
  wire                         cmd_weint;
  assign                       cmd_weint = 0;

  // Assign clock
  wire clk;
  assign clk = clk1;

  // Wires for register control
  wire regWeInt;
  wire regReInt;

  // --------------------------------------------------------------------------------------
  // External status registers
  ereg cmd(cmd_q, regBus, cmd_d, cmd_weint, regWe, regRe, clk, reset);
  defparam cmd.w    = cmd_w;
  defparam cmd.rval = cmd_r;
  defparam cmd.sgn  = cmd_s;
  defparam cmd.adr  = R_cmd;

  // --------------------------------------------------------------------------------------
  // Define register control output
  assign regWeOut = 0 | regWeInt;
  assign regReOut = 0 | regReInt;

// ----------------------------------------------------------------------------------------
//=========================================================================================

//the orignal definitions follow -> change later

//addresses mapped to        register bus
  parameter cmd_a  = R_cmd;
  parameter pAdr_a = R_pramAdr;
  parameter pDat_a = R_pramDat;
  parameter dAdr_a = R_dramAdr;
  parameter dDat_a = R_dramDat;

//memory definitions
  parameter dram_m = DramAdrWidth;  
  parameter dram_n = DramLen;
  parameter pram_m = PramAdrWidth;  
  parameter pram_n = PramLen32;
  
//-----------------------------------------------------------------------------
//DramDec to X
  wire    [dram_m-1:0]  dramAdr;                            // -> Dram
  wire          [15:0]  dramDatDec2Dram;                    // -> Dram
  wire          [15:0]  dramDatDram2Dec;                    // <- Dram
  wire                  dramWe;                             // -> Dram
  wire          [15:0]  x16Adr;                             // <- Agu
  wire          [15:0]  x16DatOut;                          // -> Agu
  wire          [15:0]  x16DatIn;                           // <- Agu
  wire                  x16DatWe;                           // <- Agu
  wire                  x16DatRe;                           // <- Agu

//PramDec to X
  wire    [pram_m-1:0]  pramAdr;                            // -> Pram
  wire          [31:0]  pramDatIn;                          // -> Pram
  wire                  pramWe;                             // -> Pram
  wire                  pramRe;                             // -> Pram
  wire          [13:0]  pramAdrPcu;                         // <- Pcu

//DEC to X
  wire          [ 3:0]  aluCmd;                             // -> Alu, also serves as shift for mul
  wire          [ 3:0]  aguCmd;                             // -> Agu
  wire          [ 1:0]  pcuCmd;                             // -> Pcu
  wire          [ 1:0]  aguMux;                             // -> Mux
  wire          [ 1:0]  pcuMux;                             // -> Mux
  wire          [ 1:0]  regMux;                             // -> Mux
  wire          [15:0]  dec2Mux;                            // -> Mux
  wire          [10:0]  aguAdr;                             // -> Agu
  wire          [ 1:0]  aguModReg;                          // -> Agu
  wire          [ 1:0]  regXSel;                            // -> Reg
  wire          [ 1:0]  regYSel;                            // -> Reg
  wire                  regWe2;                             // -> Reg
  wire                  flagAluWe;                          // -> Reg
  wire                  flagShWe;                           // -> Reg
  wire                  halt;                               // -> misc
  wire                  x16Wait;                            // <- DramDec
  wire          [ 7:0]  flagIn;                             // <- Reg
  wire          [23:0]  pramPcu2Dec;                        // <- Pcu
  wire          [15:0]  mux2Dec;                            // <- Mux

//REG to X
  wire          [15:0]  regX;                               // -> Alu   
  wire          [15:0]  regY;                               // -> Alu  
  wire          [15:0]  reg2Mux;                            // -> Mux
  wire          [15:0]  mux2Reg;                            // <- Mux
  wire          [ 7:0]  flagAlu2Reg;                        // <- Alu

//ALU to X
  wire          [15:0]  alu2Mux;                            // -> Mux
  
//Mul to X
  wire          [15:0]  mul2Mux;                            // -> Mux
  
//PCU to X
  wire          [15:0]  pcu2Mux;                            // -> Mux
  wire          [15:0]  mux2Pcu;                            // <- Mux
  wire          [31:0]  pramDat;                            // <- Pram

//AGU to X
  wire          [15:0]  agu2Mux;                            // -> Mux
  wire          [15:0]  mux2Agu;                            // <- Mux

//Interrupt lines
  reg                   irqCmdD;
  wire                  irqInt;
  
//-----------------------------------------------------------------------------
//Internal + external interupt
  always @(posedge clk)
    irqCmdD <= regWe & (regBus==R_cmd);
  assign irqInt = irqExt | irqCmdD;

//-----------------------------------------------------------------------------
//include DRAM
  spRam1 dram(dramDatDram2Dec, dramDatDec2Dram, dramAdr, dramWe, clk);
  defparam dram.m = dram_m;                                 //address width
  defparam dram.n = dram_n;                                 //number of elements
  defparam dram.w = 16;                                     //data width

  x16DramDec x16DramDec(.dramAdr(dramAdr),
                        .dramDatOut(dramDatDec2Dram),
                        .dramDatIn(dramDatDram2Dec),
                        .dramWe(dramWe), 
                        .x16Adr(x16Adr),  
                        .x16DatOut(x16DatOut), 
                        .x16DatIn(x16DatIn), 
                        .x16We(x16DatWe),  
                        .x16Re(x16DatRe), 
                        .x16Wait(x16Wait),
                        .regBus(regBus), 
                        .regWeOut(regWeInt), 
                        .regReOut(regReInt), 
                        .regWeIn(regWe), 
                        .regReIn(regRe),
                        .clk(clk)
                        );
  defparam x16DramDec.m      = dram_m;
  defparam x16DramDec.adrAdr = dAdr_a;
  defparam x16DramDec.adrDat = dDat_a;

//-----------------------------------------------------------------------------
//include PRAM
//spRam1(q, d, adr, we, clk);
  spRam1 pram(pramDat, pramDatIn, pramAdr, pramWe, clk);
  defparam pram.m = pram_m;                                 //address width
  defparam pram.n = pram_n;                                 //number of elements
  defparam pram.w = 32;                                     //data width

  x16PramDec x16PramDec(.pramAdr(pramAdr),
                        .pramDat(pramDatIn),
                        .pramWe(pramWe),
                        .x16Adr(pramAdrPcu),
                        .regBus(regBus),
                        .regWe(regWe),
                        .clk(clk)
                        );
  defparam x16PramDec.m      = pram_m;
  defparam x16PramDec.adrAdr = pAdr_a;
  defparam x16PramDec.adrDat = pDat_a;

//-----------------------------------------------------------------------------
//decoder 
  x16Dec x16Dec(.aluCmd(aluCmd),
                .aguCmd(aguCmd),
                .pcuCmd(pcuCmd),
                .aguMux(aguMux),
                .pcuMux(pcuMux),
                .regMux(regMux),
                .decImm(dec2Mux),
                .aguAdr(aguAdr),
                .aguModReg(aguModReg),
                .regXSel(regXSel),
                .regYSel(regYSel),
                .regWe(regWe2),
                .flagAluWe(flagAluWe),
                .flagShWe(flagShWe),
                .haltOut(halt),
                .haltIn(x16Wait), 
                .flagIn(flagIn), 
                .pramDat(pramPcu2Dec),
                .muxIn(mux2Dec),
                .reg_bus(regBus),
                .reg_we(regWe),
                .reg_re(regRe),
                .irqExt(irqInt),
                .clk(clk),
                .reset(reset)
                );

//REG
  x16Reg x16Reg(.regX(regX), 
                .regY(regY),
                .flagOut(flagIn), 
                .muxOut(reg2Mux),
                .muxIn(mux2Reg),
                .regXSel(regXSel),
                .regYSel(regYSel),
                .flagIn(flagAlu2Reg),
                .regWe(regWe2),
                .flagAluWe(flagAluWe),
                .flagShWe(flagShWe),
                .clk(clk),
                .reset(reset)
                );

//ALU
  x16Alu x16Alu(.c(alu2Mux), 
                .f(flagAlu2Reg), 
                .x(regX), 
                .y(regY), 
                .sel(aluCmd)
                );

//Mul
  x16Mul x16Mul(.c(mul2Mux), 
                .x(regX), 
                .y(regY), 
                .sh(aluCmd),
                .clk(clk)
                );

//PCU
  x16Pcu x16Pcu(.padr(pramAdrPcu),
                .pdatOut(pramPcu2Dec),
                .muxOut(pcu2Mux),
                .pdatIn(pramDat),
                .muxIn(mux2Pcu),
                .cmd(pcuCmd),
                .halt(halt),
                .clk(clk),
                .reset(reset)
                );

//AGU
  x16Agu x16Agu(.datAdr(x16Adr),
                .datIn(x16DatOut),
                .datOut(x16DatIn),
                .datWe(x16DatWe),
                .datRe(x16DatRe),
                .muxOut(agu2Mux),
                .muxIn(mux2Agu),
                .decAdrIn(aguAdr),
                .cmd(aguCmd),
                .modReg(aguModReg),
                .halt(halt),
                .clk(clk),
                .reset(reset)
                );

//MUX
  x16Mux x16Mux(.aguOut(mux2Agu),
                .decOut(mux2Dec),
                .pcuOut(mux2Pcu),
                .regOut(mux2Reg),
                .aluIn(alu2Mux),
                .aguIn(agu2Mux),
                .decIn(dec2Mux),
                .pcuIn(pcu2Mux),
                .regIn(reg2Mux),
                .mulIn(mul2Mux),
                .aguSel(aguMux),
                .pcuSel(pcuMux),
                .regSel(regMux)
                );
    
endmodule
//-----------------------------------------------------------------------------

    
