//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16Reg
//
// PURPOSE:
//   Register file
//
// INPUT:
//   muxIn:       Input from MUX
//   regXSel:     Select x output to ALU
//   regYSel:     Select y output to ALU
//   flagIn:      Flag input from ALU
//   flagAluWe:   Flag write enable
//   flagShWe:    Shift flag write enable
//
// OUTPUT:
//   regX:        Output x to ALU
//   regY:        Output y to ALU
//   flagOut:     Flag output to ALU
//   muxOut:      Output to MUX
//
// DESCRIPTION:
//   Store X16 register R0-R3 and the flags. The registers are updated according
//   to the input from the muxer and regYSel. At the same time, outputs regX/Y
//   are passed to the ALU.
//
// HISTORY:
//   11-Nov-06, Dirk Sommer
//     Initial version
//
//====================================================================================

// ----------------------------------------------------------------------------
module x16Reg(regX, 
              regY,
              flagOut, 
              muxOut,
              muxIn,
              regXSel,
              regYSel,
              flagIn,
              regWe,
              flagAluWe,
              flagShWe,
              clk, reset);

  // --------------------------------------------------------------------------
  // Define inputs and outputs
  output      [15:0]  regX;                           // ALU input x
  output      [15:0]  regY;                           // ALU input y
  output      [ 7:0]  flagOut;                        // 
  output      [15:0]  muxOut;
  input       [15:0]  muxIn;
  input       [ 1:0]  regXSel;
  input       [ 1:0]  regYSel;
  input       [ 7:0]  flagIn;                         // Decoder flag input
  input               regWe;
  input               flagAluWe;
  input               flagShWe;
  input               clk, reset;

  // --------------------------------------------------------------------------
  // Include definitions
  `include "def_X16.v"

  // --------------------------------------------------------------------------
  // Define registers and wires
  reg       [15:0]  R0, R1, R2, R3;
  reg       [ 7:0]  FL;
  reg       [15:0]  regX, regY; 

  // --------------------------------------------------------------------------
  // Output to mux
  assign muxOut = regX;

  // Flag output - possibly use bypass to allow immediate evaluation
  assign flagOut[0]   = 1;                                  // no condition
  assign flagOut[3:1] = flagAluWe ? flagIn[3:1] : FL[3:1];  // GT,EQ,LT
  assign flagOut[7:5] = flagAluWe ? flagIn[7:5] : FL[7:5];  // LE,NE,GE
  assign flagOut[4]   = flagShWe  ? flagIn[4]   : FL[4];    // SH
  
  // -------------------------------------------------------------------------------------
  // Set ALU output X, IM is possible
  always @(*) begin
    case (regXSel)
      R0i: regX = R0;
      R1i: regX = R1;
      R2i: regX = R2;
      R3i: regX = R3;
    endcase

    case (regYSel)
      R0i: regY = R0;
      R1i: regY = R1;
      R2i: regY = R2;
      R3i: regY = R3;
    endcase
  end
  
  // -------------------------------------------------------------------------------------
  // Update reg = R0-R3
  always @(posedge clk) begin

    // Reset flags 
    if (reset)
      FL <= 8'b00000001;
  
    // Normal flag operation
    else begin
      if (flagAluWe) begin
        FL[3:1] <= flagIn[3:1];
        FL[7:5] <= flagIn[7:5];
      end

      if (flagShWe) begin
        FL[4] <= flagIn[4];
      end
    end // if (reset) .. else

    // Update registers
    if (regWe) begin
      case (regYSel)
        R0i: R0 <= muxIn;
        R1i: R1 <= muxIn;
        R2i: R2 <= muxIn;
        R3i: R3 <= muxIn;
      endcase
    end
    
  end // always @(posedge clk)
  
endmodule
// ----------------------------------------------------------------------------

    
