//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16Mux
//
// PURPOSE:
//   Distribute data between X16 modules
//
// INPUT:
//   aluIn:     Input from ALU
//   aguIn:     Input from AGC
//   decIn:     Input from DEC
//   pcuIn:     Input from PCU
//   regIn      Input from REG
//   aguSel:    Selector for output to AGU
//   pcuSel:    Selector for output to PCU
//   regSel:    Selector for output to REG
//
// OUTPUT:
//   aguOut:    Output to AGU
//   decOut:    Output to DEC
//   pcuOut:    Output to PCU
//   regOut:    Output to REG
//
// DESCRIPTION:
//   Multiplex inputs to outputs:
//     aguOut  = (decIn, pcuIn, regIn)
//     decOut  = (regIn)
//     pcuOut  = (aguIn, decIn, regIn)
//     regOut  = (aluIn, aguIn, decIn, mulIn)
//
// HISTORY:
//   11-Nov-06, Dirk Sommer
//     Initial version
//
//====================================================================================

//========================================================================================
module x16Mux(aguOut,
              decOut,
              pcuOut,
              regOut,
              aluIn,
              aguIn,
              decIn,
              pcuIn,
              regIn,
              mulIn,
              aguSel,
              pcuSel,
              regSel
              );

  // -------------------------------------------------------------------------------------
  // Declare ports
  output    [15:0]  aguOut;
  output    [15:0]  decOut;
  output    [15:0]  pcuOut;
  output    [15:0]  regOut;
  input     [15:0]  aluIn;
  input     [15:0]  aguIn;
  input     [15:0]  decIn;
  input     [15:0]  pcuIn;
  input     [15:0]  regIn;
  input     [15:0]  mulIn;
  input     [ 1:0]  aguSel;
  input     [ 1:0]  pcuSel;
  input     [ 1:0]  regSel;

  // -------------------------------------------------------------------------------------
  // Output register declaration
  reg       [15:0]  aguOut;
  reg       [15:0]  pcuOut;
  reg       [15:0]  regOut;

  // -------------------------------------------------------------------------------------
  // Include X16 definitions
  `include "def_X16.v"

  // -------------------------------------------------------------------------------------
  // Assign decOut directly, no selection needed currently
  assign decOut = regIn;

  // Define all multiplexors
  always @(*) begin
    
    // Mux AGU
    case (aguSel)
      Dec2Agu:  aguOut = decIn;
      Pcu2Agu:  aguOut = pcuIn;
      Reg2Agu:  aguOut = regIn;
      default:  aguOut = regIn;  // Just something
    endcase
    
    // Mux PCU
    case (pcuSel)
      Agu2Pcu:  pcuOut = aguIn;
      Dec2Pcu:  pcuOut = decIn;
      Reg2Pcu:  pcuOut = regIn;
      default:  pcuOut = regIn;  // Just something
    endcase

    // Mux REG
    case (regSel)
      Alu2Reg:  regOut = aluIn;
      Agu2Reg:  regOut = aguIn;
      Dec2Reg:  regOut = decIn;
      Mul2Reg:  regOut = mulIn;
    endcase

  end //always
endmodule
//========================================================================================

    
