//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16Dec
//
// PURPOSE:
//   Instruction decoder
//
// INPUT:
//   waitIn:      Halt the system, e.g. because read take more than a cycle
//   flagIn:      Flags input from x16Dec
//   pramDat:     Pram data
//   lc0:         Indicate that LC = 0
//   reg_bus:     Register register input
//   regWe/Re:    Corresponding read/write signals
//   irqExt:      External IRQ line
//
// OUTPUT:
//   aluCmd:      Command for ALU
//   aguCmd:      Command for AGU -> nop/read/write/call/ret/set
//   pcuCmd:      Command for PCU -> normal/loop/wait/jump
//   decImm:      Immediate value (extract from pramDat)
//   regXSel/Y:   Select register for ALU input X/Y
//   selRegIn:    Select input for register 
//
// DESCRIPTION:
//   The X16-DEC accepts ALU flags and the a wait flag, PRAM data etc and
//   decodes the commands for ALU, AGU, PCU, etc..., combines internal halt
//   conditions with external halt signal, continues from halt upon IRQ etc.
//
// HISTORY:
//   11-11-06, Dirk Sommer
//     Initial version
//
//====================================================================================

//-------------------------------------------------------------------------------
module x16Dec(aluCmd,
              aguCmd,
              pcuCmd,
              aguMux,
              pcuMux,
              regMux,
              decImm,
              aguAdr,
              aguModReg,
              regXSel,
              regYSel,
              regWe,
              flagAluWe,
              flagShWe,
              haltOut,
              haltIn, 
              flagIn, 
              pramDat,
              muxIn,
              reg_bus,
              reg_we,
              reg_re,
              irqExt,
              clk, reset);

  // -------------------------------------------------------------------------------
  // define ports
  output              [ 3:0]  aluCmd;                       // select ALU operation
  output              [ 3:0]  aguCmd;                       // select AGU command
  output              [ 1:0]  pcuCmd;                       // select PCU command
  output              [ 1:0]  aguMux;                       // select AGU input
  output              [ 1:0]  pcuMux;                       // select PCU input
  output              [ 1:0]  regMux;                       // select REG input
  output              [15:0]  decImm;                       // decoder immediate value
  output              [10:0]  aguAdr;                       // decoder immediate address value
  output              [ 1:0]  aguModReg;                    // modifier/register for AGU commands
  output              [ 1:0]  regXSel;                      // select register for ALU input X
  output              [ 1:0]  regYSel;                      // select register for ALU input Y
  output                      regWe;                        // write enable for register 
  output                      flagAluWe;                    // write enable ALU flags
  output                      flagShWe;                     // write enable for SHIFT flag
  output                      haltOut;                      // halt the system
  input                       haltIn;                       // halt input for state machine
  input               [ 7:0]  flagIn;                       // flag input
  input               [23:0]  pramDat;                      // pram data input
  input               [15:0]  muxIn;                        // input from multiplexor
  inout               [15:0]  reg_bus;                      // register bus
  input                       reg_we;                       // register bus write enable
  input                       reg_re;                       // register bus read enable
  input                       irqExt;                       // external IRQ line
  input                       clk, reset;

  // -------------------------------------------------------------------------------
  // use registers for output
  reg                 [ 3:0]  aluCmd;
  reg                 [ 3:0]  aguCmd;
  reg                 [ 1:0]  pcuCmd;
  reg                 [ 1:0]  aguMux;
  reg                 [ 1:0]  pcuMux;
  reg                 [ 1:0]  regMux;
  reg                 [15:0]  decImm;   
  reg                 [10:0]  aguAdr;   
  reg                 [ 1:0]  aguModReg;   
  reg                 [ 1:0]  regXSel;  
  reg                 [ 1:0]  regYSel;
  reg                         regWe;   
  reg                         flagAluWe;
  reg                         flagShWe; 

  // -------------------------------------------------------------------------------
  // define a few internal wires and registers
  wire                [23:0]  p;
  
  reg                 [ 3:0]  state;
  reg                         haltFlag;
  wire                [ 1:0]  p0b21p1, p0b21m1;
  reg                         flagCnd;
  reg                         resetD;
  
  wire                        halt;

  reg                 [ 7:0]  WC;
  reg                         waitCntWeImm, waitCntWeReg;
  reg                 [ 7:0]  LC;
  reg                         loopWeImm, loopWeReg, loopDown, loop0;
  
  // -------------------------------------------------------------------------------
  // include x16 definitions
  `include "def_X16.v"

  // -------------------------------------------------------------------------------
  // define halt of the system
  assign halt       = (haltFlag & ~irqExt & ~(WC==1)) | haltIn;
  assign haltOut    = halt;

  // -------------------------------------------------------------------------------
  // some assignments for simplification
  assign p        = pramDat;
  
  // +1/-1 for register definition
  assign p0b21p1  = p[7:6]+1;
  assign p0b21m1  = p[7:6]-1;

  // evaluate condition, could use indexing, flagCnd = flagIn[p1[7:5]] 
  always @(flagIn or p[7:5])
    case (p[7:5])
      NCi: flagCnd = flagIn[0];
      CYi: flagCnd = flagIn[1];
      EQi: flagCnd = flagIn[2];
      LTi: flagCnd = flagIn[3];
      SHi: flagCnd = flagIn[4];
      LEi: flagCnd = flagIn[5];
      NEi: flagCnd = flagIn[6];
      GEi: flagCnd = flagIn[7];
    endcase
  
  // -------------------------------------------------------------------------------
  // state machine, fetch/decode/execute...
  always @(posedge clk) begin
    resetD  <= reset;                                       // make sure this reset is longer than the
                                                            // reset of the registers                                                            

    // -------------------------------------------------------------------------------
    // wait counter
    if (resetD) begin
      WC           <= 0;
      waitCntWeImm <= 0;
      waitCntWeReg <= 0;
    end 
      
    else begin
      if (waitCntWeImm) begin                               // init wait counter
        WC           <= decImm[7:0];                        // with immediate value
        waitCntWeImm <= 0;
      end else
      if (waitCntWeReg) begin                               // init wait counter with
        WC           <= muxIn[7:0];                         // register R3
        waitCntWeReg <= 0;
      end else
      if (WC!=0) begin                                      // count down to 0
        WC <= WC-1;
      end      
    end  

    // -------------------------------------------------------------------------------
    // loop counter - enable later
    if (resetD) begin
      loopWeImm <= 0;
      loopWeReg <= 0;
    end 
    
    else if (~halt) begin
      if (loopWeImm) begin                               // init loop counter
        LC        <= decImm[7:0];                        // with immediate value
        loopWeImm <= 0;
      end else
      if (loopWeReg) begin                               // init loop counter with
        LC        <= muxIn[7:0];                         // register R3
        loopWeReg <= 0;
      end else
      if (loopDown) begin                                 // count down to 0
        LC        <= LC-1;
        loopDown  <= 0;
      end
      loop0  <= (LC==0);
    end  

    // -------------------------------------------------------------------------------
    // synchronous reset of the state machine  
    if (resetD) begin
      state     <= DecJump;
      aguCmd    <= AguNop;
      pcuCmd    <= PcuJmpX1;                                // jump by 4 for proper load
      regWe     <= 0;                                       // no register write
      flagAluWe <= 0;                                       // no flag write
      flagShWe  <= 0;
      haltFlag  <= 1;
    end else

    // -------------------------------------------------------------------------------
    // run the FSM only if halt==0
    if (~halt) begin
      
      //-------------------------------------------------------------------------------
      // first of all, some default settings (might be overwritten later)
      haltFlag      <= 0;                                   // no halt
      regWe         <= 0;                                   // no register write
      flagAluWe     <= 0;                                   // no flag write
      flagShWe      <= 0;                                   // -> unless overwritten later
      aguCmd        <= AguNop;
      pcuCmd        <= PcuFetch;
      state         <= DecDecode;                           // normally stay in decode state
          
      // -----------------------------------------------------------------------------  
      // there is a bit of performance gain if some of the instructions are processed
      // outside the if (state==Decode) branch.

      // pcuMux
      if (p[4:0]==0 & p[7:5]==X16Ret)
        pcuMux <= Agu2Pcu;
      else if (p[4:0]==0 & p[7:5]==X16Clr)
        pcuMux <= Reg2Pcu;
      else
        pcuMux <= Dec2Pcu;
          
      // aluCmd, regYSel
      if (p[0]==0) begin
        regYSel  <= p[7:6];
      
        // 2 operands                                                                         
        if (p[1]==0) begin                                                                 
          aluCmd <= {1'b0, p[4:2]};                                                        
        end                                                                                
                                                                                           
        // 1 operand                                                                        
        else begin                                                                         
          aluCmd   <= {1'b1, p[5:3]};                                                       
        end                                                                                
      end                                                                                  
                                                                                           
      // 16 bit command                                                                       
      else begin                                                                           
        regYSel  <= p[15:14];                                                               
        aluCmd   <= p[11:8];                                                                
      end                                                                                  
   
      // -----------------------------------------------------------------------------  
      // start with normal decoding
      if (state==DecDecode) begin
      
      // -----------------------------------------------------------------------------  
      // 8 bit commands  
        if (p[0]==0) begin

          if (p[1]==1'b0) begin

            // special commands
            if (p[4:2]==3'b000) begin

              // halt          
              if (p[7:5]==X16Hlt) begin
                haltFlag <= 1;
              end else

              // loop          
              if (p[7:5]==X16Lop & ~loop0) begin                // if LC!=0
                pcuCmd  <= PcuJmpR;
                decImm  <= {{8{p[15]}}, p[15:8]};               // signed extension
                state   <= DecJump;                             // use normal jump
                loopDown  <= 1;
              end else

              // wtc - wait a certain number of cycles
              if (p[7:5]==X16Wtc) begin
                decImm  <= {8'h00, p[15:8]};                    // load wait register
                regXSel <= R3i;
                haltFlag <= 1;
                if (p[15:8] != 0)
                  waitCntWeImm <= 1;
                else
                  waitCntWeReg <= 1;
              end else 

              // call
              if (p[7:5]==X16Cal) begin
                pcuCmd  <= PcuJmpA;
                aguCmd  <= AguCall1;
                decImm  <= p[23:8];
                state   <= DecCall;
              end else 

              // return
              if (p[7:5]==X16Ret) begin
                pcuCmd  <= PcuJmpA;
                aguCmd  <= AguRet;
                state   <= DecJump;
              end else 

              // load LC
              if (p[7:5]==X16Llc) begin                
                decImm  <= {8'h00, p[15:8]};
                if (p[15:8] != 0)
                  loopWeImm <= 1;
                else begin
                  loopWeReg <= 1;                
                  regXSel   <= R3i;
                end
              end

              // load PC=R3, push current PC
              if (p[7:5]==X16Clr) begin
                pcuCmd  <= PcuJmpA;
                aguCmd  <= AguCall1;
                regXSel <= R3i;
                state   <= DecCall;
              end
            end else

            //---------------------------------------------------------------------------
            // load 16 bit value, all registers
            if (p[4:2]==3'b001) begin
              decImm    <= p[23:8];          
              if (p[5]==0) begin
                regMux    <= Dec2Reg;
                regWe     <= 1;
              end else begin
                aguCmd    <= AguSetReg;
                aguMux    <= Dec2Agu;
                aguModReg <= p[7:6];
              end
            end else

            // ALU commands with 2 operands
            begin
              if (p[5]==0)
                regXSel <= p0b21p1;                 // use Rx = Ry+1 for p0[5]=0        
              else
                regXSel <= p0b21m1;                 // use Rx = Ry-1 for p0[5]=1        
              regMux    <= Alu2Reg;                 // input from ALU                   
              regWe     <= 1;                       // write enable for register        
              if (p[4:2]!=3'b010)                   // if not mov                       
                flagAluWe <= 1;                     //   set ALU flags                  
            end //end ALU commands with 2 operands

          end else //if (p[1]==1'b0)

          // ---------------------------------------------------------------------------
          // commands for p[1:0] = 2'b10

          // ALU commands with 1 operand
          begin
            if (p[2]==0) begin
              regXSel   <= p[7:6];      // select input register for ALU
              regMux    <= Alu2Reg;
              regWe     <= 1;
              if (p[5]==0)
                flagAluWe <= 1;
              else
                flagShWe  <= 1;          
            end else

            // mrp/mwp
            begin
              aguModReg  <= p[5:4];
              // mrp
              if (p[3]==0) begin
                aguCmd   <= AguRdPtr;
                regWe    <= 1;
                regMux   <= Agu2Reg;
              // mwp
              end else begin
                aguCmd   <= AguWrPtr;
                regXSel  <= p[7:6];
                aguMux   <= Reg2Agu;
              end
            end 

          end // (p[1]==0) ... else
        end else //if (p[0]==0), end of 8 bit command decoding

        // ---------------------------------------------------------------------------
        // if (p[0]==0) ... else, 16 bit commands
        begin

          // mrd/mwd
          if (p[1]==1) begin
            aguAdr <= p[13:3];

            // mrd
            if (p[2]==0) begin
              aguCmd   <= AguRdImm;
              regMux   <= Agu2Reg;
              regWe    <= 1;
            end else

            // mwd
            begin
              aguCmd  <= AguWrImm;
              aguMux  <= Reg2Agu;
              regXSel <= p[15:14];
            end
          end else

          // if (p[1]==1) ... else
          begin

            // ldv
            if (p[2]==1) begin
            
              // load R0-R3
              if (p[3]==0) begin
                decImm      <= {{6{p[13]}}, p[13:4]};
                regMux      <= Dec2Reg;
                regWe       <= 1;
              end else
              
              // load AGU register  
              begin
                if (p[15:14]==MOi)
                  decImm    <= {{6{p[13]}}, p[13:4]};
                else
                  decImm    <= {6'b000000, p[13:4]};
                aguMux      <= Dec2Agu;
                aguCmd      <= AguSetReg;
                aguModReg   <= p[15:14];
              end
            end else

            // if (p[2]==1) ... else -> all the following commands are conditional
            begin
              if (p[4:3] == 2'b11) begin

                // mov Rz -> Ry  
                if (p[11:8]==4'b0000) begin
                  aguCmd    <= AguGetReg;
                  aguModReg <= p[13:12];
                  regMux    <= Agu2Reg;
                  if (flagCnd)
                    regWe     <= 1;
                end else

                // mov Ry -> Rz
                if (p[11:8]==4'b0001) begin
                  regXSel   <= p[13:12];
                  if (flagCnd)
                    aguCmd  <= AguSetReg;
                  aguModReg <= p[15:14];
                  aguMux    <= Reg2Agu;
                end else

                // normal ALU commands
                begin //if (p[11:8] != 0001) & (p[11:8] != 0000)
                  regXSel   <= p[13:12];
                  regMux    <= Alu2Reg;
                  if (flagCnd)
                    regWe   <= 1;
                    
                  if (p[7:5]==3'b000) begin
                    
                    if (flagCnd)
                      if (p[11:10]==2'b11)
                        flagShWe <= 1;
                      else
                        if (p[11:8]!=4'b0010)    // if not mov                  
                          flagAluWe <= 1;        //   set ALU flags             
                  end
                end 
              end else //if (p[4:3] == 2'b11)

              // mul
              if (p[4:3]==2'b01) begin
                  regMux    <= Mul2Reg;
                  regXSel   <= p[13:12];                                                               
                  if (flagCnd)
                    regWe   <= 1;
              end else

              // gtr
              if (p[4:3]==2'b10) begin
                decImm    <= {{8{p[15]}}, p[15:8]};
                if (flagCnd) begin
                  state     <= DecJump;
                  pcuCmd    <= PcuJmpR;
                end
              end else

              // gta  
              begin //if (p[4:3]==2'b00)
                decImm    <= p[23:8];
                if (flagCnd) begin
                  state     <= DecJump;
                  pcuCmd    <= PcuJmpA;
                end
              end 

            end //if (p[2]==1) ... else if (flagCnd)
          end //if (p[1]==1) ... else
        end //if (p[0]==0) ... else, 16 bit commands 
      end else //if (state==DecDecode)
      
      // -------------------------------------------------------------------------------
      // jump state
      if (state==DecJump) begin
        state   <= DecJumpWait1;
        pcuCmd  <= PcuJmpX1;
      end else
      
      // -------------------------------------------------------------------------------
      // call state
      if (state==DecCall) begin
        state   <= DecJumpWait1;
        pcuCmd  <= PcuJmpX1;
        aguCmd  <= AguCall2;
        aguMux  <= Pcu2Agu;
      end else

      // -------------------------------------------------------------------------------
      // wait for jump to finish (1)
      if (state==DecJumpWait1) begin
        state   <= DecJumpWait2;
        pcuCmd  <= PcuFetch;
      end else

      // -------------------------------------------------------------------------------
      // wait for jump to finish (2)
      if (state==DecJumpWait2) begin
        state   <= DecDecode;
        pcuCmd  <= PcuFetch;
      end

    end //if (reset==1) ... else
  end //always
endmodule

// -------------------------------------------------------------------------------

   
