//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16Alu
//
// PURPOSE:
//   ALU processing
//
// INPUT:
//   x:         Data input x
//   y:         Data input y
//   sel:       Operation selection
//
// OUTPUT:
//   c:         Reset of operation
//   f:         Flag update
//
// DESCRIPTION:
//   The X16-ALU executes all operations between data registers (R0..R3)
//   of the X16 processor. The following operations are available
//
//     AluMov : c = x
//     AluAdd : c = y + x
//     AluSub : c = y - x
//     AluAnd : c = y & x
//     AluOor : c = y | x
//     AluXor : c = y ^ x
//     AluInc : c = x + 1
//     AluDec : c = x - 1
//     AluNot : c = ~x
//     AluNeg : c = -x
//     AluSrl : c = x >> 1 (unsigned shift)
//     AluSra : c = x >> 1 (signed shift)
//     AluShl : c = x << 1
//     AluRor : c = rotate right
//  
//    The input are extended by 0 before the operation starts in order to allow
//    calculation of the carry flag for add/sub/inc/dec operations. Nonetheless,
//    only the lower 16 bit are used for the output c. The flags are:
//      
//      GE : >=0
//      NE : ~=0
//      LE : <=0
//      SH : Shift flag (bit 15 or bit 0, depending or direction)
//      LT : <0
//      EQ : =0
//      CY : Carry flag
//
//    Note that there is no >0 flag.
//
//    This module covers only the arithmetic operation itself, there is no 
//    output registering of the results.
//
// HISTORY:
//   11-11-06, Dirk Sommer
//     Initial version
//
//====================================================================================

module x16Alu(c, f, x, y, sel);

  output    [15:0] c;
  output    [ 7:0] f;
  input     [15:0] x,y;
  input     [ 3:0] sel;

  // ------------------------------------------------------------------------------
  // Define wires for extended inputs
  wire      [16:0] xe, ye;

  // Define output
  reg       [16:0] ce;
  wire      [15:0] c;  

  // ------------------------------------------------------------------------------
  // Define wires for flags
  wire      eq, ge, gt, lt, ne, le, sh, cy; 

  // ------------------------------------------------------------------------------
  // Include definitions
  `include "def_X16.v"

  // ------------------------------------------------------------------------------
  // Unsigned extension of inputs
  assign xe = {1'b0, x};
  assign ye = {1'b0, y};

  // ------------------------------------------------------------------------------
  // ALU operation
  always @(*) begin
    case (sel)
      AluMov : ce = xe;
      AluAdd : ce = ye + xe;
      AluSub : ce = ye - xe;
      AluAnd : ce = ye & xe;
      AluOor : ce = ye | xe;
      AluXor : ce = ye ^ xe;
      AluInc : ce = xe + 1;
      AluDec : ce = xe - 1;
      AluNot : ce = ~xe;
      AluNeg : ce = -xe;
      AluSrl : ce = {1'b0, 1'b0,     xe[15:1]};
      AluSra : ce = {1'b0, xe[15],   xe[15:1]};
      AluShl : ce = {1'b0, xe[14:0],     1'b0};
      AluRor : ce = {1'b0, xe[0],    xe[15:1]};
      default: ce = xe;
    endcase     
  end

  // Get 16 bit output
  assign c = ce[15:0];

  // ------------------------------------------------------------------------------
  // Normal Alu flags  
  assign ge = c[15]==0;
  assign eq = c==0;
  assign gt = ge & ne;
  assign lt = ~ge;
  assign ne = ~eq;
  assign le = ~gt;

  // Shift flag
  assign sh = (sel==AluShl) ? x[15] : x[0];

  // Carry flag
  assign cy = ce[16];
  
  // Assign flag output
  assign f = {ge, ne, le, sh, lt, eq, cy, 1'b1};  

endmodule
//==============================================================================

    
