//====================================================================================
//  Copyright (C) BAY9, 2011
//====================================================================================
//
// MODULE:
//   x16Agu
//
// PURPOSE:
//   Address generation for X16
//
// INPUT:
//   datIn:     Data input
//   muxIn:     Input from MUX
//   decAdrIn:  Immediate address input from decoder
//   cmd:       Command (read/write/load/call/ret)
//   modReg:    Modifier for PR/PW, or register to read/write (PR/PW/MO,SP)
//   halt:      Halt operation
//
// OUTPUT:
//   datAdr:    Address output
//   datOut:    Data output
//   datWe:     Data write enable
//   datRe:     Indicate the DSP to wait for data
//   muxOut:    Output to MUX
//
// DESCRIPTION:
//   The X16-AGU addresses the memory and the memory-mapped register bus.
//   It contains 4 registers:
//     PR: read pointer
//     PW: write pointer
//     MO: modifier register
//     SP: stack pointer
//
//   The AGU supports the following commands at its input:
//     AguCall1:
//       SP = SP-1;
//     AguCall2:
//       Dram(SP) = muxIn (from PCU);
//     AguRet:
//       muxOut = dram(SP);
//       SP = SP+1;
//     AguRdImm:
//       muxOut = Dram(decAdrIn);
//     AguWrImm:
//       Dram(decAdrIn) = muxIn;
//     AguRdPtr:
//       muxOut = Dram(PR);
//       PR = PR += X; (X=0/1/MO/-1, depending on input modReg)
//     AguWrPtr:
//       Dram(PW) = muxIn;
//       PW = PW += X; (X=0/1/MO/-1, depending on input modReg)
//     AguSetReg:
//       Reg = muxIn; (Reg = PR/PW/MO/SP, depending on input modReg)
//     AguGetReg:
//       muxOut = Reg; (Reg = PR/PW/MO/SP, depending on input modReg)
//
// HISTORY:
//   11-Nov-06, Dirk Sommer
//     Initial version
//
//====================================================================================

//====================================================================================
module x16Agu(datAdr,
              datIn,
              datOut,
              datWe,
              datRe,
              muxOut,
              muxIn,
              decAdrIn,
              cmd,
              modReg,
              halt,
              clk, reset);

  // -----------------------------------------------------------------------------------
  // Input / output definition
  output    [15:0]  datAdr;   
  output    [15:0]  datOut;   
  input     [15:0]  datIn;    
  output            datWe;    
  output            datRe;    
  output    [15:0]  muxOut;   
  input     [15:0]  muxIn;    
  input     [10:0]  decAdrIn; 
  input     [ 3:0]  cmd;      
  input     [ 1:0]  modReg;   
  input             halt;     
  input             clk, reset;
  
  // -------------------------------------------------------------------------------------
  // Register / wire definition

  // Define some outputs as registers
  reg       [15:0]  datAdr;
  reg               datWe;
  reg               datRe;
  reg       [15:0]  muxOut;

  // Internal registers
  reg       [15:0]  PR, PW, SP, MO;
  
  // Internal wires, some defined as registers for simplicity
  reg       [15:0]  regInp;
  reg               regWe;
  reg       [ 1:0]  regSel;
  reg       [15:0]  addIn2;
  reg       [ 1:0]  addIn2Sel;
  reg               regInpSel;
  reg       [ 2:0]  muxOutSel;
  wire      [15:0]  addOut;
  
  // --------------------------------------------------------------------------------------
  // Includes
  `include "def_X16.v"

  // --------------------------------------------------------------------------------------
  // Combinatorial logic - evaluate commands
  always @(*) begin

    // SP--
    if (cmd==AguCall1) begin
      muxOutSel = 3'bx;
      regSel    = SPi;
      addIn2Sel = AguModM1;
      regInpSel = AguLoadAdd;
      regWe     = 1;
      datWe     = 0;
      datRe     = 0;
    end else
      
    // Write data from mux to mem, use SP
    if (cmd==AguCall2) begin
      muxOutSel = 3'bx;
      regSel    = SPi;
      addIn2Sel = 2'bx;
      regInpSel = 1'bx;
      regWe     = 0;
      datWe     = 1;
      datRe     = 0;
    end else
      
    // Read data from mem to mux, use SP, SP++
    if (cmd==AguRet) begin
      muxOutSel = DATi;
      regSel    = SPi;
      addIn2Sel = AguModP1;
      regInpSel = AguLoadAdd;
      regWe     = 1;
      datWe     = 0;
      datRe     = 1;
    end else

    // Read data from mem to mux, use decoder immediate
    if (cmd==AguRdImm) begin
      muxOutSel = DATi;
      regSel    = IMi;
      addIn2Sel = 2'bx;
      regInpSel = 1'bx;
      regWe     = 0;
      datWe     = 0;
      datRe     = 1;
    end else

    // Write data from mux to mem, use decoder immediate
    if (cmd==AguWrImm) begin
      muxOutSel = 3'bx;
      regSel    = IMi;
      addIn2Sel = 2'bx;
      regInpSel = 1'bx;
      regWe     = 0;
      datWe     = 1;
      datRe     = 0;
    end else

    // Read data from mem to mux, use pointer
    if (cmd==AguRdPtr) begin
      muxOutSel = DATi;
      regSel    = PRi;
      addIn2Sel = modReg;
      regInpSel = AguLoadAdd;
      regWe     = 1;
      datWe     = 0;
      datRe     = 1;
    end else

    // Write data from mux to mem
    if (cmd==AguWrPtr) begin
      muxOutSel = 3'bx;
      regSel    = PWi;
      addIn2Sel = modReg;
      regInpSel = AguLoadAdd;
      regWe     = 1;
      datWe     = 1;
      datRe     = 0;
    end else

    // Set a register with data from the muxer
    if (cmd==AguSetReg) begin
      muxOutSel = 3'bx;
      regSel    = modReg;
      addIn2Sel = 2'bx;
      regInpSel = AguLoadMux;
      regWe     = 1;
      datWe     = 0;
      datRe     = 0;
    end else

    // Get a register, write it to the muxer
    if (cmd==AguGetReg) begin
      muxOutSel = {1'b0, modReg};
      regSel    = 2'bx;
      addIn2Sel = 2'bx;
      regInpSel = 1'bx;
      regWe     = 0;
      datWe     = 0;
      datRe     = 0;
    end else
    
    // In all other cases, simply disable writing, don't care about the rest
    begin
      muxOutSel = 3'bx;
      regSel    = 2'bx;
      addIn2Sel = 2'bx;
      regInpSel = 1'bx;
      regWe     = 0;
      datWe     = 0;
      datRe     = 0;
    end  

  // --------------------------------------------------------------------------------------
  // Multiplexors

    // Mux for address
    case (regSel)
      PRi:  datAdr = PR;
      PWi:  datAdr = PW;
      SPi:  datAdr = SP;
      IMi:  datAdr = {5'b0, decAdrIn};
    endcase

    // Mux for adder input 1 -> not needed, simply use datAdr

    // Mux for adder input 2
    case(addIn2Sel)
      AguMod0:   addIn2 =  0;
      AguModP1:  addIn2 =  1;
      AguModMO:  addIn2 = MO;
      AguModM1:  addIn2 = -1;
    endcase

    // Mux for register update
    case (regInpSel)
      AguLoadAdd:  regInp = addOut;
      AguLoadMux:  regInp = muxIn;
    endcase

    // Mux for mux output
    case (muxOutSel)
      PRi:      muxOut = PR;
      PWi:      muxOut = PW;
      SPi:      muxOut = SP;
      MOi:      muxOut = MO;
      DATi:     muxOut = datIn;
      default:  muxOut = datIn; // Just something
    endcase

  end //always @(*)

  // --------------------------------------------------------------------------------------
  // Define adder, using the address as adder input
  assign addOut = datAdr + addIn2;

  // Pass through data
  assign datOut = muxIn;

  // --------------------------------------------------------------------------------------
  // State machine update
  always @(posedge clk) begin

    // Write update for registers
    if (regWe & ~halt) begin
      case (regSel)
        PRi:  PR <= regInp;
        PWi:  PW <= regInp;
        SPi:  SP <= regInp;
        MOi:  MO <= regInp;
      endcase     
    end    

  end //always
endmodule
//====================================================================================
