//=========================================================================================
//  Copyright (C) BAY9, 2011
//=========================================================================================
//
// MODULE:
//   ereg
//
// PURPOSE:
//   Externally accessible register inside a module
//
// INPUT:
//   d:      Internal data input 
//   weint:  Internal write enable
//   we:     External write enable
//   re:     External read enable
//
// OUTPUT:
//   q:      Internal output
// 
// INOUT:
//   dq:     Exernal input/output bus interface
//
// PARAMETER:
//   w:      Internal register bit width
//   rval:   Reset value
//   adr:    Register address
//   sgn:    Sign extension flag
//   
// DESCRIPTION:
//   The EREG register serves as configuration register of a module. It
//   can be written or read externally via a bus access or directly from
//   inside the module. When reading from regBus, the output is  signed
//   extended or not depending on sgn parameter if the register width is
//   less than 16 bit. Typically, it is written from outside and and read
//   from inside the module, but it can also serve as module status, when
//   updated internally.
//
//   The operation is as follows:
//     Writing via bus access:
//       if (we & (adr==dq))
//         q <= dq  (in the next cycle)
//     Reading via bus:
//       if (re & (adr==dq))
//         dq <= q  (in the next cycle)
//     Writing from inside the module:
//       if (weint)
//         q <= d
//     Reading from inside the module:
//       q can always be read
//
// HISTORY:
//   07-Oct-2004, Dirk Sommer:
//     Initial version
//
//==============================================================================

module ereg(q, dq, d, weint, we, re, clk, reset);

  // ---------------------------------------------------------------------------
  // Define parameters
  parameter w     = 16;                 // Data width
  parameter rval  = 0;                  // Reset value
  parameter adr   = 0;                  // Address specification
  parameter sgn   = 0;                  // Signed/unsigned output

  // ---------------------------------------------------------------------------
  // Define inputs and outputs
  output [w-1:0] q;                     // Internal data output
  inout  [ 15:0] dq;                    // External input/output
  input  [w-1:0] d;                     // Internal data input

  input  weint;                         // Internal write enable
  input  we;                            // External write enable
  input  re;                            // External read enable

  input  clk;                           // Clock
  input  reset;                         // Reset all internal states
  
  // ---------------------------------------------------------------------------
  // Define internal states
  reg    [w-1:0]   q;                   // The output is a register
  reg              we2;                 // Delayed write enable
  reg              re2;                 // Delayed read enable

  // This is a trick to get length 16 working
  wire   [16:0]  dq2;

  // ---------------------------------------------------------------------------
  // Enable external write/read
  always @(posedge clk) begin
    we2 <= we & (dq==adr);
    re2 <= re & (dq==adr);
  end

  // Operation reset and writing
  always @(posedge clk)

    if (reset) begin                    // Asynchronous reset
      q <= rval;
      
    end else begin                      // Normal write operation
      if (we2)
        q <= dq[w-1:0];
      else
        if (weint)
          q <= d;
    end

  // Tristate input/output
  assign dq2 = (sgn==0) ? {{(17-w){1'b0}},   q} :
                          {{(17-w){q[w-1]}}, q} ;
  assign dq  = (re2 & ~reset) ? dq2[15:0] : {16{1'bz}};
    
endmodule
//==============================================================================
