function cfm = msgVrfCfgTxPaLutReq(id, ampFac, phOffDeg)
% ==============================================================================
%  Copyright (C) BAY9, 2014
% ==============================================================================
%
% Syntax:
%   cfm = msgVrfCfgTxPaLutReq(id, bypass)
%
% Inputs:
%   id            - Target identifier
%   ampFac        - Vector 1x513 amplitude factors, [0..1]
%   phOffDeg      - Vector 1x513 phase offset in degrees, [-180..+180]
%
% Outputs:
%   cfm           - [MsgIdVrf.CfgTxPaLutCfm status]
%
% Description:
%   This function sets up the power amplifier look-up tables. The 513
%   amplitude factors and phase offset correspond to input data amplitudes
%   of [0, 64, 128, ..., 32768]. They are converted to fix point as
%     ampFacFxp = ampFac * 2^15
%     phOffFxp  = phOffDeg / 360 * 2^16
%   and written to the VRF core in an interleaved fashion.
% 
%   The function checks the confirm MsgId and the return status.
%
% ==============================================================================

  % ----------------------------------------------------------------------------
  % Global defintions
  global MsgIdVrf;

  % Define message core
  id.core = 'vrf';
  
  % ----------------------------------------------------------------------------
  % Check input range
  if (any(ampFac < 0) || any(ampFac > 1))
    find(ampFac - 1 > 0)
    keyboard
    error('Amplitude factor > 1 is not allowed');
  end
  if (any(phOffDeg < -180) || any(phOffDeg > 180))
    error('Angle deviation must be in the range [-180..180]');
  end
  
  % Calculate 16 bit input
  ampFacFxp = round(ampFac*2^15);
  phOffFxp  = round(phOffDeg/360*2^16);
  
  % Paranoia check, 32768 -> 32767
  ampFacFxp = ampFacFxp - (ampFacFxp == 32768);
  phOffFxp  = phOffFxp  - (phOffFxp  == 32768);

  % Resort interleaved
  lut       = [ampFacFxp; phOffFxp];
  lut       = lut(:)';
  
  % Send message
  req   = [MsgIdVrf.CfgTxPaLutReq lut];
  nCfm  = 2;
  cfm   = sendMsg(id, req, nCfm);

  % Check confirm
  if (cfm(1) ~= MsgIdVrf.CfgTxPaLutCfm)
    error('Wrong confirm message received');
  end
  
  % Check if message is supported
  if (cfm(2) ~= 0)
    fprintf('Warning:\n  -> PA modeling not supported, msg has no effect\n');
  end
  
% ==============================================================================
