function cfm = msgVrfCfgRxIqImbReq(id, ampFac, alpha)
% ==============================================================================
%  Copyright (C) BAY9, 2013
% ==============================================================================
%
% Syntax:
%   cfm = msgVrfCfgRxIqImbReq(id, ampFac, alpha)
%
% Inputs:
%   id            - Target identifier
%   ampFac        - Amplitude factor of Q-path, [0..infty]
%   alpha         - Angle in deg deviation of Q-path, [-180..180]
%
% Outputs:
%   cfm           - [MsgIdVrf.CfgRxIqImbCfm status]
%
%  Description:
%    The message calculates the parameters, a, b, c as
% 
%      kf   = sqrt(2 / (1+ampFac^2))
%      a    =  kf
%      b    =  kf*ampFac*cos(alpha)
%      c    = -kf*ampFac*sin(alpha)
%  
%   where a, b, c, are used in module iqImb tp calculate the output
%
%      (yRe)         ( 1  -ampFac*sin(alpha)) ) (xRe)   (a  c) (xRe) 
%      (   ) =  kf * (                        ) (   ) = (    ) (   )
%      (yIm)         ( 0   ampFac*cos(alpha)) ) (xIm)   (0  b) (xIm) 
%
%   i.e., the Q-path is assumed to be impaired by factor ampFac and
%   angle alpha. Factor kf is used to normalize the total power. 
%   Undistorted transmission occurs for ampFac=1, alpha=0. Factors a, b, 
%   and c are scaled by 2^14 for fixed point operation.
%
%   The function checks the confirm MsgId and the return status.
%   
% ==============================================================================

  % ----------------------------------------------------------------------------
  % Global defintions
  global MsgIdVrf;
  
  % Define message core
  id.core = 'vrf';
  
  % ----------------------------------------------------------------------------
  % Check input range
  if (ampFac < 0)
    error('Amplitude factor < 0 is not allowed');
  end
  if (alpha < -180 || alpha > 180)
    error('Angle deviation must be in the range [-180..180]');
  end
  
  % Calculate IQ imbalance factors
  kf   = sqrt(2 / (1+ampFac^2));
  aFxp = round( kf*2^14);
  bFxp = round( kf*ampFac*cos(alpha*2*pi/360)*2^14);
  cFxp = round(-kf*ampFac*sin(alpha*2*pi/360)*2^14);
  
  % Send message
  req   = [MsgIdVrf.CfgRxIqImbReq aFxp bFxp cFxp];
  nCfm  = 2;
  cfm   = sendMsg(id, req, nCfm);

  % Check confirm
  if (cfm(1) ~= MsgIdVrf.CfgRxIqImbCfm)
    error('Wrong confirm message received');
  end
  
  % Check if message is supported
  if (cfm(2) ~= 0)
    fprintf('Warning:\n  -> RX-IQ imbalance not supported, msg has no effect\n');
  end
  
% ==============================================================================
