function cfm = msgVrfCfgRxGainTblReq(id, gainDelay, gainDb, gainSel)
% ==============================================================================
%  Copyright (C) BAY9, 2013
% ==============================================================================
%
% Syntax:
%   cfm = msgVrfCfgChGainReq(id, gainDelay, gainDb, gainSel)
%
% Inputs:
%   id          - Target identifier
%   gainDelay   - Gain delay in samples
%   gainDb      - Gain values in dB
%   gainSel     - Gain pin settings to select gain values, 0..127
%
% Outputs:
%   cfm   - MsgIdVrf.CfgChGainReq
%
% Description:
%   The function sets gain delay and the RX gain table with 128 entries
%   for gainFactor (0..255, 8 bit) and gainShift(-32..18, 6 bit). Each 
%   entry corresponds to a certain 7-bit pin pattern given by gainSel.
%
%   While parameter gainDelay is a scalar of value 1..1023, gainDb and 
%   gainSel are vectors of length between 1..128. There is no need to
%   define all 128 gain values. Unused positions of the LUT will be set 
%   to 0 gain by default.
%
%   Parameter gainSel determines he pin pattern the LUT is addressed
%   twith, it is used as index of he internal vector in this function.
%   
%   Parameter gainDb refers to the ADC output RMS of the WGN of the VRF
%   IP core for with respect to the 12 bit ADC output. E.g.,  for
%   gainDb = 0, full scale noise with RMS = 2048 would result, for
%   gainDb = -20, noise RMS = 204.8. As the internal WGN level is
%   RMS(n) = 796/2^4 first calculates an offset,  that is, the gain
%   needed to reach 0 dB fullscale
%
%     gainOffsetDb = -20*log10(796/2^4/2^11)
%
%   For each input gainDb, the corresponding Gain shift + factor are then
%   calculated as
%
%     f           = 10^((gainDb+gainOffsetDb)/20)
%     gainShift   = ceil(log2(f))
%     gainFactor  = round(256*2^(log2(f) - ceil(log2(f))))
%     if (gainFactor == 256)
%       gainShift  = gainShift + 1
%       gainFactor = 128
%     end
%
%   and written to the position indicated by gainSel. Finally, the
%   gainDelay and the gainShift/gainFactor values are then sent to the
%   target via the CfgRxGainTbl message in format
%
%     Msg = [MsgId gainDelay gainShift0, gainFactor0, gainShift1, ...]
%
%   for all 128 values of gainShift/gainFactor. 
%
% ==============================================================================

  % ----------------------------------------------------------------------------
  % Global defintions
  global MsgIdVrf;

  % Define message core
  id.core = 'vrf';
  
  % ----------------------------------------------------------------------------
  % Paranoia check if length are OK
  if (length(gainDb) ~= length(gainSel))
    gainDb, gainSel
    error('Length of gainDb and gainSel not equal');
  end
  
  % Calculate gain offset with respect to 12 bit full scale AGWN noise
  % -> AWGN noise is assumed to have RMS = 796 @ 16 bit
  gainOffsetDb = -20*log10(796/2^4/2^11);

  % Preset outputs
  gainShift  = zeros(1, 128);
  gainFactor = zeros(1, 128);
    
  % For all inputs
  for k = 1:length(gainDb)
    
    % Get index of current gain to set, check for duplicate definition and range
    l = gainSel(k)+1;            % Matlab write index
    if (gainFactor(l) ~= 0)
      k, duplicateIndex = gainSel(k)
      error('Duplicate definition');
    end

    % Get floating point multiplication factor
    f = 10^((gainDb(k) + gainOffsetDb) / 20);

    % Calculate gainShift + gainFactor
    gainShift(l)  = ceil(log2(f));                          % Apply this effective shift
    gainFactor(l) = round(256*2^(log2(f) - ceil(log2(f)))); % Apply this factor
    if (gainFactor(l) == 256)                               % Paranoia range check
      gainShift(l)  = gainShift(l) + 1;
      gainFactor(l) = 128;
    end

    % Paranoia check
    if (gainShift(l)  > 18  || gainShift(l)  < -32)
      gainIndexL  = k
      gainShiftL  = gainShift(l)
      gainFactorL = gainFactor(l)
      error('Gain shift out of range');
    end
  end
  
  % Sort as gainShift(0), gainFactor(0), gainShift(1), gainFactor(1),....
  gainAll = [gainShift; gainFactor];
  gainAll = gainAll(:)';
  
  % Send message (convert
  req   = [MsgIdVrf.CfgRxGainTblReq gainDelay gainAll];
  nCfm  = 1;
  cfm   = sendMsg(id, req, nCfm);

  % Check confirm
  if (cfm(1) ~= MsgIdVrf.CfgRxGainTblCfm)
    error('Wrong confirm message received');
  end
  
% ==============================================================================
