function cfm = msgVrfCfgMultiPathReq(id, c, d)
% ==============================================================================
%  Copyright (C) BAY9, 2013
% ==============================================================================
%
% Syntax:
%   cfm = msgVrfCfgMultiPathReq(id, c, d)
%
% Inputs:
%   id    - Target identifier
%   c     - Coefficient values, complex with amp <= 2, row vector
%   d     - Filter CF delay, [0..29], must be unique, row vector
%
% Outputs:
%   cfm   - [MsgIdVrf.CfgMultiPathCfm status]
%
% Description:
%   Calculate the fixed point coefficient values and write the message of
%   up to 10 coefficients the system as:
%
%     [MsgId, nC, cRe0, cIm0, d0, cRe1, cIm1, ..., d(nC-1)]
%
%   The magnitude of the complex coefficients must not exceed 2. The message
%   can contain an arbitrary number of coefficients between 1..10
%
%   The function checks the confirm MsgId and the return status.
%
% ==============================================================================

  % ----------------------------------------------------------------------------
  % Global defintions
  global MsgIdVrf;

  % Define message core
  id.core = 'vrf';
  
  % ----------------------------------------------------------------------------
  % Check for number of coefficients
  if (length(c) < 1 || length(c) > 10)
    c
    error('Number of coefficients out of range 1..10');
  end
  
  % Check for number of coefficient values + position are the same
  if (length(c) ~= length(d))
    c
    d
    error('Number of coefficients values and coefficient positions is not the same');
  end
  
  % Check for magnitude
  if (any(abs(c) > 2))
    c
    error('Coefficient value exceeds 2');
  end
  
  % Check for duplicate positions
  if (any(diff(sort(d)) == 0))
    d
    error('Duplicate coefficient position defined');
  end
  
  % Check for positions outside allowed range
  if (any(d < 0) || any(d > 29))
    d
    error('Coefficient position out of range 0..29');
  end
  
  % Convert to Q13 + interleave to compose message
  cRe = round(real(c*2^13));
  cIm = round(imag(c*2^13));
  cReImDel = [cRe; cIm; d];
  cReImDel = cReImDel(:)';
  
  % Send message
  req   = [MsgIdVrf.CfgMultiPathReq length(c) cReImDel];
  nCfm  = 2;
  cfm   = sendMsg(id, req, nCfm);

  % Check confirm
  if (cfm(1) ~= MsgIdVrf.CfgMultiPathCfm)
    error('Wrong confirm message received');
  end
  
  % Check if message is supported
  if (cfm(2) ~= 0)
    fprintf('Warning:\n  -> Multipath not supported, msg has no effect\n');
  end
  
% ==============================================================================
