function cfm = msgVrfCfgFreqOffReq(id, clkOffPpm, fC, fS)
% ==============================================================================
%  Copyright (C) BAY9, 2013
% ==============================================================================
%
% Syntax:
%   cfm = msgVrfCfgFreqOffReq(id, clkOffPpm, fC, fS)
%
% Inputs:
%   id          - Target identifier
%   clkOffPpm   - Clock offset in ppm
%   fC          - Carrier frequency
%   fS          - Sampling frequency 
%
% Outputs:
%   cfm         - [MsgIdVrf.CfgFreqOffCfm status]
%
% Description:
%   Based on the offset in PPM, the carrier frequency, and the sample frequency,
%   calculate the relative frequency offset as
%   
%     freqOffRel = clkOffPpm/1e6 * fC / fS
%
%   convert it to 48 bit fixed point, and pass it to VRF core.
%
%   The function checks the confirm MsgId and the return status.
%
%
% ==============================================================================

  % ----------------------------------------------------------------------------
  % Global defintions
  global MsgIdVrf;

  % Define message core
  id.core = 'vrf';
  
  % ----------------------------------------------------------------------------
  % Calculate frequency offset relative to sample frequency
  freqOffRel = clkOffPpm/1e6 * fC / fS; 

  % Check input range
  if (freqOffRel < -0.5 || freqOffRel >= 0.5)
    freqOffRel
    error('Relative frequency offset must be between [-0.5..+0.5]');
  end

  % Convert to fixed point
  freqOff48   = round(freqOffRel*2^48);
  freqOffLo16 = mod(floor(freqOff48 / 2^0 ), 2^16);
  freqOffMi16 = mod(floor(freqOff48 / 2^16), 2^16);
  freqOffHi16 = mod(floor(freqOff48 / 2^32), 2^16);

  % Send message
  req   = [MsgIdVrf.CfgFreqOffReq, freqOffLo16, freqOffMi16, freqOffHi16];
  nCfm  = 2;
  cfm   = sendMsg(id, req, nCfm);
  
  % Debug: check confirm
  if (cfm(1) ~= MsgIdVrf.CfgFreqOffCfm)
    fprintf('Error sending Frequency offset config request\n');
    pause
  end

  % Check if message is supported
  if (cfm(2) ~= 0)
    fprintf('Warning:\n  -> Frequency offset not supported, msg has no effect\n');
  end

% ==============================================================================
