function cfm = msgVrfCfgClkOffReq(id, clkOffPpm, byPass)
% ==============================================================================
%  Copyright (C) BAY9, 2013
% ==============================================================================
%
% Syntax:
%   cfm = msgVrfCfgClkOffReq(id, clkOffPpm, byPass)
%
% Inputs:
%   id          - Target identifier
%   clkOffPpm   - Clock offset in PPM, [-1000..+1000]
%   byPass      - 0/1, 1 = bypass the filter, i.e use h = [0 ... 0 1 0... 0]
%
% Outputs:
%   cfm         - [MsgIdVrf.CfgClkOffCfm status]
%
% Description:
%   Set the clock offset in PPM using
%
%     req = [MsgIdVrf.CfgFreqOff, cO32Lo16, cO32Mi16, cO32Hi16, byPass]
%
%   and read the confirm message. The message assumes the clock at the
%   transmitter being the reference, while the clock at the receive has
%   an offset of clkOffPpm. The time step per 32 samples is calculated as
%   
%     cO32 = (1/(1+clkOffPpm/1e6)-1) * 32  \approx -clkOffPpm/1e6
%
%   so the time step is negative if clkOffPpm > 0, i.e. if the sampling frequency
%   is higher at the RX than at TX.
%
%   The function checks the confirm MsgId and the return status.
%
% ==============================================================================

  % ----------------------------------------------------------------------------
  % Global defintions
  global MsgIdVrf;

  % Define message core
  id.core = 'vrf';
  
  % ----------------------------------------------------------------------------
  % Check input range
  if (abs(clkOffPpm) > 1000)
    clkOffPpm
    error('Clock offset must be < 1000 ppm');
  end
%   if (abs(clkOffPpm) > 100)
%     fprintf('\nWarning:\n  -> Clock offset > 100ppm results in reduced accuracy\n');
%   end
%   if (clkOffPpm > 0)
%     fprintf('\nWarning:\n');
%     fprintf('  -> Clock offset > 0, make sure to set DAC rate < 1 to avoid FIFO overflow\n');
%   end
  if (byPass == 1 && clkOffPpm ~= 0)
    fprintf('\nWarning:\n');
    fprintf('  -> Bypass set, clock offset ~= 0 ignored\n');
  end

  % Calculate time step offset using the same definition as for frequency offset
  %  -> frequency is increased by factor 1+clkOffPpm/1e6
  %  -> sample period is 1/(1+clkOffPpm/1e6)
  cO = 1/(1+clkOffPpm/1e6)-1;
  
  % Convert to fixed point for 32 samples
  cO32Bit48 = round(cO*32*2^48);
  cO32Lo16  = mod(floor(cO32Bit48 / 2^0 ), 2^16);
  cO32Mi16  = mod(floor(cO32Bit48 / 2^16), 2^16);
  cO32Hi16  = mod(floor(cO32Bit48 / 2^32), 2^16);

  % Send message
  req   = [MsgIdVrf.CfgClkOffReq, cO32Lo16, cO32Mi16, cO32Hi16, byPass];
  nCfm  = 2;
  cfm   = sendMsg(id, req, nCfm);
  
  % Debug: check confirm
  if (cfm(1) ~= MsgIdVrf.CfgClkOffCfm)
    fprintf('Error sending time offset config request\n');
    pause
  end

  % Check if message is supported
  if (cfm(2) ~= 0)
    fprintf('Warning:\n  -> Clock offset not supported, msg has no effect\n');
  end

% ==============================================================================
