function cfm = msgVrfCfgChGainReq(id, snr)
% ==============================================================================
%  Copyright (C) BAY9, 2013
% ==============================================================================
%
% Syntax:
%   cfm = msgVrfCfgChGainReq(id, snr)
%
% Inputs:
%   id    - Target identifier
%   snr   - Time domain SNR
%
% Outputs:
%   cfm   - MsgIdVrf.CfgChGainReq
%
% Description:
%   Assuming input data RMS = 3276.8, and WGN RMS = 796 (based on complex valued
%   data), gain factor and gain shift are calculated as
%
%     f           = 10^(snr/20) * (WGN RMS / input data RMS)
%     gainShift   = ceil(log2(f))
%     gainFactor  = round(256*2^(log2(f) - ceil(log2(f))))
%     if (gainFactor == 256)
%       gainShift  = gainShift + 1
%       gainFactor = 128
%     end
%
%   With given assumptions for default input + noise RMS, SNR setting from
%   roughly -180..+120 dB are possibly, though very low SNRs do not make sense
%   as there is no signal anymore.
%
% ==============================================================================

  % ----------------------------------------------------------------------------
  % Global defintions
  global MsgIdVrf;
  
  % Define message core
  id.core = 'vrf';
  
  % ----------------------------------------------------------------------------
  % Calculate gain shift/factor + check for error
  wgnRms      = 796;
  inpRms      = 3276.8;
  f           = 10^(snr/20) * (wgnRms / inpRms);        % Float factor
  gainShift   = ceil(log2(f));                          % Apply this effective shift
  gainFactor  = round(256*2^(log2(f) - ceil(log2(f)))); % Apply this factor
  if (gainFactor == 256)                                % Paranoia range check
    gainShift  = gainShift + 1;
    gainFactor = 128;
  end

  if (gainShift  > 18  || gainShift  < -32 || ...
      gainFactor > 255 || gainFactor < 128)
    gainShift
    gainFactor
    error('Gain shift and/or factor out of range');
  end
  
  % Send message
  req   = [MsgIdVrf.CfgChGainReq gainShift gainFactor];
  nCfm  = 1;
  cfm   = sendMsg(id, req, nCfm);

  % Check confirm
  if (cfm(1) ~= MsgIdVrf.CfgChGainCfm)
    error('Wrong confirm message received');
  end
  
% ==============================================================================
